<?php
namespace wcf\acp\form;
use wcf\form\AbstractForm;
use wcf\system\exception\UserInputException;
use wcf\system\registry\RegistryHandler;
use wcf\system\style\StyleCompiler;
use wcf\system\style\StyleHandler;
use wcf\system\WCF;
use wcf\util\FileUtil;
use wcf\util\StringUtil;

/**
 * Shows the form input for global style values.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @package	WoltLabSuite\Core\Acp\Form
 */
class StyleGlobalValuesForm extends AbstractForm {
	/**
	 * @inheritDoc
	 */
	public $activeMenuItem = 'wcf.acp.menu.link.style.globalValues';
	
	/**
	 * global SCSS styles
	 * @var string
	 */
	public $styles = '';

	/**
	 * current scroll offset before submitting the form
	 * @var integer
	 */
	public $stylesScrollOffset = 0;
	
	/**
	 * @var string
	 */
	public $styleTestFileDir;
	
	/**
	 * @inheritDoc
	 */
	public function readFormParameters() {
		parent::readFormParameters();
		
		if (isset($_POST['styles'])) {
			$this->styles = StringUtil::unifyNewlines(StringUtil::trim($_POST['styles']));
		}
		if (isset($_POST['stylesScrollOffset'])) {
			$this->stylesScrollOffset = intval($_POST['stylesScrollOffset']);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		if (empty($_POST)) {
			$this->styles = (string)RegistryHandler::getInstance()->get('com.woltlab.wcf', StyleCompiler::REGISTRY_GLOBAL_VALUES);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function validate() {
		parent::validate();
		
		$tmpFile = FileUtil::getTemporaryFilename('styleGlobalValues_', '.scss');
		file_put_contents($tmpFile, $this->styles);
		
		try {
			// Due to performance issues we can only compile the default style and check, 
			// whether there are syntax issues. 
			$defaultStyle = StyleHandler::getInstance()->getDefaultStyle();
			if ($defaultStyle !== null) {
				$this->styleTestFileDir = FileUtil::getTemporaryFilename('style_');
				FileUtil::makePath($this->styleTestFileDir);
				
				$errorMessage = StyleCompiler::getInstance()->testStyle($this->styleTestFileDir, $defaultStyle->styleName, $defaultStyle->apiVersion, $defaultStyle->imagePath, $defaultStyle->getVariables(), $tmpFile);
				
				if ($errorMessage !== null) {
					rmdir($this->styleTestFileDir);
					
					throw new UserInputException('styles', [
						'message' => str_replace($tmpFile, '', $errorMessage->getMessage()),
					]);
				}
			}
		}
		finally {
			unlink($tmpFile);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function save() {
		parent::save();
		
		if (empty($this->styles)) {
			RegistryHandler::getInstance()->delete('com.woltlab.wcf', StyleCompiler::REGISTRY_GLOBAL_VALUES);
			
			if (file_exists(WCF_DIR . StyleCompiler::FILE_GLOBAL_VALUES)) {
				unlink(WCF_DIR . StyleCompiler::FILE_GLOBAL_VALUES);
			}
		}
		else {
			RegistryHandler::getInstance()->set('com.woltlab.wcf', StyleCompiler::REGISTRY_GLOBAL_VALUES, $this->styles);
			
			file_put_contents(WCF_DIR . StyleCompiler::FILE_GLOBAL_VALUES, "/*\n\n  DO NOT EDIT THIS FILE!\n\n  dynamic global SCSS values, generated at ".date('r', TIME_NOW)."\n\n*/\n\n" . $this->styles);
		}
		
		// call saved event
		$this->saved();
		
		// reset stylesheets
		StyleHandler::resetStylesheets(false);
		
		// save compiled style
		$defaultStyle = StyleHandler::getInstance()->getDefaultStyle();
		if ($defaultStyle !== null && $this->styleTestFileDir && file_exists($this->styleTestFileDir . '/style.css') && file_exists($this->styleTestFileDir . '/style-rtl.css')) {
			$styleFilename = StyleCompiler::getFilenameForStyle($defaultStyle);
			rename($this->styleTestFileDir . '/style.css', $styleFilename . '.css');
			rename($this->styleTestFileDir . '/style-rtl.css', $styleFilename . '-rtl.css');
			
			rmdir($this->styleTestFileDir);
		}
		
		WCF::getTPL()->assign('success', true);
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign([
			'styles' => $this->styles,
			'stylesScrollOffset' => $this->stylesScrollOffset,
		]);
	}
}
