<?php
namespace wcf\page;
use wcf\system\application\ApplicationHandler;
use wcf\system\exception\IllegalLinkException;
use wcf\system\request\LinkHandler;
use wcf\system\Regex;
use wcf\system\WCF;
use wcf\util\HeaderUtil;
use wcf\util\StringUtil;

/**
 * Dereferer page
 *
 * @author	Sascha Greuel, Sonnenspeer
 * @copyright	2014-2015 Sonnenspeer
 * @license	GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 */
class DerefererPage extends AbstractPage {
	/**
	 * @var string
	 */
	public $externalLink = '';
	
	/**
	 * @var string
	 */
	protected $illegalChars = '[^\x0-\x2C\x2E\x2F\x3A-\x40\x5B-\x60\x7B-\x7F]+';
	
	/**
	 * @see	\wcf\page\AbstractPage::assignVariables()
	 */
	public function readParameters() {
		parent::readParameters();
		
		if(isset($_REQUEST['ref'])) $this->externalLink = $this->unescape(base64_decode(rawurldecode(StringUtil::trim($_REQUEST['ref']))));
		
		$regex = new Regex('
		(?<!\B|"|\'|=|/|\]|,|\?|\.)
		(?:						# hostname
			(?:ftp|https?)://'.$this->illegalChars.'(?:\.'.$this->illegalChars.')*
			|
			www\.(?:'.$this->illegalChars.'\.)+
			(?:[a-z]{2,4}(?=\b))
		)
		
		(?::\d+)?					# port
		
		(?:
			/
			[^!.,?;"\'<>()\[\]{}\s]*
			(?:
				[!.,?;(){}]+ [^!.,?;"\'<>()\[\]{}\s]+
			)*
		)?', Regex::IGNORE_WHITESPACE | Regex::CASE_INSENSITIVE);
		
		if (empty($this->externalLink) || !$regex->match($this->externalLink)) {
			throw new IllegalLinkException();
		}
		
		if (ApplicationHandler::getInstance()->isInternalURL($this->externalLink) || !StringUtil::executeWordFilter($this->externalLink, EXTERNALLINKS_WHITELIST)) {
			HeaderUtil::redirect($this->externalLink);
			exit;
		}
	}
	
	/**
	 * @see	\wcf\page\AbstractPage::assignVariables()
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign(array(
			'externalLink' => $this->externalLink,
			'referedURI' => (!empty($_SERVER['HTTP_REFERER']) ? StringUtil::encodeHTML($_SERVER['HTTP_REFERER']) : LinkHandler::getInstance()->getLink())
		));
	}
	
	/**
	 * JS unescape() equivalent
	 * 
	 * @param	string		$encodedStr
	 * @return	string
	 */
	private function unescape($encodedStr) {
		$decodedStr = '';
		$pos = 0;
		$len = mb_strlen($encodedStr);
		
		while ($pos < $len) {
			$charAt = mb_substr($encodedStr, $pos, 1);
			
			if ($charAt == '%') {
				$pos += 1;
				
				if (mb_substr($encodedStr, $pos, 1) == 'u') {
					// we've got an unicode character
					$pos += 1;
					
					$unicodeHexVal = mb_substr($encodedStr, $pos, 4);
					$unicode = hexdec($unicodeHexVal);
					$entity = '&#' . $unicode . ';';
					$decodedStr .= utf8_encode($entity);
					
					$pos += 4;
				}
				else {
					// we've got an escaped ascii character
					$hexVal = mb_substr($encodedStr, $pos, 2);
					$unicode = hexdec($hexVal);
					$entity = '&#' . $unicode . ';';
					$decodedStr .= utf8_encode($entity);
					
					$pos += 2;
				}
			}
			else {
				$decodedStr .= $charAt;
				$pos++;
			}
		}
		
		return htmlspecialchars_decode(StringUtil::convertEncoding('HTML-ENTITIES', 'UTF-8', $decodedStr));
	}
}
