<?php

namespace wcf\system\option;

use wcf\data\multirank\Multirank;
use wcf\data\option\Option;
use wcf\data\user\option\UserOption;
use wcf\system\exception\UserInputException;
use wcf\system\WCF;

/**
 * Option type for multiranks
 *
 * @author        Olaf Braun
 * @copyright     2013-2019 Olaf Braun - Software Development
 * @license       WBB-Elite.de License <https://lizenz.wbb-elite.de/lizenz.html>
 * @package       de.wbb-elite.multirank
 * @category      Several rank graphics
 */
class MultirankOptionType extends CheckboxesOptionType {
	
	private $mustSelect = [];
	
	/**
	 * @inheritdoc
	 */
	public function getFormElement(Option $option, $value) {
		if ($option instanceof UserOption && $option->user) {
			$value = explode("\n", $value);
			WCF::getTPL()->assign([
				'option'        => $option,
				'selectOptions' => $this->getMultirankList($option->user->getUserID()),
				'value'         => $value,
				'mustSelect'    => $this->mustSelect
			]);
			
			return WCF::getTPL()->fetch('multirankselect');
		}
		
		return '';
	}
	
	/**
	 * @inheritdoc
	 */
	public function validate(Option $option, $newValue) {
		if (!is_array($newValue)) $newValue = [];
		if (!empty($newValue)) {
			if ($option instanceof UserOption && $option->user) {
				$userRanks = $this->getMultirankList($option->user->getUserID());
				foreach ($newValue as $rank) {
					if (!isset($userRanks[$rank])) {
						throw new UserInputException($option->optionName, 'empty');
					}
				}
			}
		}
	}
	
	/**
	 * Return the ranks for user option
	 *
	 * @param integer $userID
	 *
	 * @return array
	 */
	private function getMultirankList($userID) {
		$userRanks = Multirank::getRanks($userID);
		$result = [];
		if (!empty($this->mustSelect)) {
			$this->mustSelect = [];
		}
		foreach ($userRanks as $userRank) {
			if (mb_strlen($userRank->rankImage) > 0) {
				$result[$userRank->rankID] = $userRank->getImage();
			} else {
				$result[$userRank->rankID] = '<span class="badge label ' . $userRank->cssClassName . '">' . WCF::getLanguage()->get($userRank->rankTitle) . '</span>';
			}
			/** @noinspection PhpUndefinedFieldInspection */
			if ($userRank->multirankOption) {
				$this->mustSelect[] = $userRank->rankID;
			}
		}
		
		return $result;
	}
}
