<?php

namespace wcf\system\multirank;

use wcf\data\object\type\ObjectTypeCache;
use wcf\system\SingletonFactory;

/**
 * Managed the multirank show objects
 *
 * @author        Olaf Braun
 * @copyright     2013-2019 Olaf Braun - Software Development
 * @license       WBB-Elite.de License <https://lizenz.wbb-elite.de/lizenz.html>
 * @package       de.wbb-elite.multirank
 * @category      Several rank graphics
 */
class MultirankHandler extends SingletonFactory {
	
	/**
	 * mapes the names of the multriank object types to the object type ids
	 *
	 * @var    array<integer>
	 */
	protected $objectTypeIDs = [];
	
	/**
	 * list of multriank object types
	 *
	 * @var    array<\wcf\data\object\type>
	 */
	protected $objectTypes = [];
	
	/**
	 * Returns a list of multriank object types.
	 *
	 * @return    array<\wcf\data\object\type>
	 */
	public function getMultriankTypes() {
		return $this->objectTypes;
	}
	
	/**
	 * Returns a title of a multriank object type
	 *
	 * @param    integer $objectTypeID
	 *
	 * @return    string
	 */
	public function getMultriankTitle($objectTypeID) {
		if (isset($this->objectTypeIDs[$objectTypeID])) {
			$object = $this->getObjectTypeByName($this->objectTypeIDs[$objectTypeID]);
			
			return $object->getProcessor()->getTitle();
		}
		
		return "";
	}
	
	/**
	 * Gets the object type with the given name.
	 *
	 * @param    string $objectType
	 *
	 * @return    \wcf\data\object\type\ObjectType
	 */
	public function getObjectTypeByName($objectType) {
		if (isset($this->objectTypes[$objectType])) {
			return $this->objectTypes[$objectType];
		}
		
		return null;
	}
	
	/**
	 * Gets the object type with the given id.
	 *
	 * @param    integer $objectTypeID
	 *
	 * @return    \wcf\data\object\type\ObjectType
	 */
	public function getObjectType($objectTypeID) {
		if (isset($this->objectTypeIDs[$objectTypeID])) {
			return $this->getObjectTypeByName($this->objectTypeIDs[$objectTypeID]);
		}
		
		return null;
	}
	
	/**
	 * @inheritdoc
	 */
	protected function init() {
		$this->objectTypes = ObjectTypeCache::getInstance()->getObjectTypes('de.wbb-elite.multirank');
		foreach ($this->objectTypes as $objectType) {
			$this->objectTypeIDs[$objectType->objectTypeID] = $objectType->objectType;
		}
	}
}
