<?php

namespace wcf\system\event\listener;

use wcf\data\AbstractDatabaseObjectAction;
use wcf\data\multirank\Multirank;
use wcf\data\user\rank\UserRank;
use wcf\data\user\rank\UserRankList;
use wcf\data\user\User;
use wcf\data\user\UserAction;
use wcf\data\user\UserEditor;
use wcf\data\user\UserProfile;
use wcf\system\WCF;

/**
 * A event that update all ranks for user buy change user group and or update his points
 *
 * @author        Olaf Braun
 * @copyright     2013-2019 Olaf Braun - Software Development
 * @license       WBB-Elite.de License <https://lizenz.wbb-elite.de/lizenz.html>
 * @package       de.wbb-elite.multirank
 * @category      Several rank graphics
 */
class UpdateUserMultirankListener implements IParameterizedEventListener {
	
	/**
	 * List off all user ranks
	 *
	 * @var array
	 */
	private $userRanks = [];
	
	/**
	 * @inheritdoc
	 */
	public function execute($eventObj, $className, $eventName, array &$parameters) {
		if (MODULE_MULTIRANK) {
			$ranks = new UserRankList();
			$ranks->getConditionBuilder()->add('user_rank.multirankOption = ?', [1]);
			$ranks->sqlJoins = "LEFT JOIN	wcf" . WCF_N . "_user_group user_group
				ON		(user_group.groupID = user_rank.groupID)";
			$ranks->sqlOrderBy = "user_group.priority DESC, user_rank.requiredPoints DESC, user_rank.requiredGender DESC";
			$ranks->readObjects();
			$this->userRanks = $ranks->getObjects();
			/** @var AbstractDatabaseObjectAction $eventObj */
			if (count($this->userRanks) > 0) {
				if ($eventObj->getActionName() == "create") {
					if ($eventObj instanceof UserAction) {
						$return = $eventObj->getReturnValues();
						$user = $return["returnValues"];
						$this->updateUserRank($user->userID);
					} else {
						$parameters = $eventObj->getParameters();
						$data = $parameters["data"];
						if (isset($data["userID"]) && !empty($data["userID"])) {
							$this->updateUserRank($data["userID"]);
						}
					}
				} else if ($eventObj->getActionName() == "addToGroups") {
					foreach ($eventObj->getObjectIDs() as $userID) {
						$this->updateUserRank($userID);
					}
				} else if ($eventObj->getActionName() == "update" && $eventObj instanceof UserAction) {
					$return = $eventObj->getReturnValues();
					$parameters = $eventObj->getParameters();
					$optionID = User::getUserOptionID("multirankIDs");
					$rankIDs = null;
					if (isset($parameters["options"]) && isset($parameters["options"][$optionID])) {
						$rankIDs = explode("\n", $parameters["options"][$optionID]);
					}
					if (isset($return["objectIDs"])) {
						foreach ($return["objectIDs"] as $userID) {
							$this->updateUserRank($userID, $rankIDs);
						}
					}
				} else if ($eventObj->getActionName() == "addMember") {
					$parameters = $eventObj->getParameters();
					if (isset($parameters["user"])) {
						$user = $parameters["user"];
						$this->updateUserRank($user->userID);
					}
				} else if ($eventObj->getActionName() == "enable" || $eventObj->getActionName() == "updateUserRank") {
					foreach ($eventObj->getObjects() as $user) {
						/** @var UserEditor|User|UserProfile $user */
						$this->updateUserRank($user->userID);
					}
				}
			}
		}
	}
	
	/**
	 * Update the user ranks for this user
	 *
	 * @param    integer $userID
	 * @param array      $rankIDs
	 *
	 * @throws \wcf\system\database\DatabaseException
	 */
	public function updateUserRank($userID, $rankIDs = null) {
		$user = new User($userID);
		if ($rankIDs == null) {
			$ranks = $this->getUserRank($user);
		} else {
			$ranks = $rankIDs;
		}
		if (!is_array($ranks)) {
			$ranks = [];
		}
		$canUseRanks = Multirank::getRanks($userID);
		$rankList = new UserRankList();
		$rankList->readObjects();
		$allRanks = $rankList->getObjects();
		foreach ($ranks as $key => $rankID) {
			if (!MULTIRANK_ONLY_ONE_PER_GROUP && !isset($canUseRanks[$rankID])) {
				unset($ranks[$key]);
			} else {
				if (!isset($allRanks[$rankID])) {
					unset($ranks[$key]);
					continue;
				}
				$userRank = $allRanks[$rankID];
				if (!in_array($userRank->groupID, $user->getGroupIDs())) {
					unset($ranks[$key]);
					continue;
				}
				if (!MULTIRANK_ONLY_ONE_PER_GROUP && !($user->activityPoints >= $userRank->requiredPoints)) {
					unset($ranks[$key]);
					continue;
				}
				if (!($user->gender && ($userRank->requiredGender == 0 || $userRank->requiredGender == $user->gender)) && !($user->gender == 0 && $userRank->requiredGender == 0)) {
					unset($ranks[$key]);
					continue;
				}
			}
		}
		foreach ($this->userRanks as $userRank) {
			if (!in_array($userRank->rankID, $ranks) && in_array($userRank->groupID, $user->getGroupIDs()) && $user->activityPoints >= $userRank->requiredPoints) {
				if (($user->gender && ($userRank->requiredGender == 0 || $userRank->requiredGender == $user->gender)) || ($user->gender == 0 && $userRank->requiredGender == 0)) {
					$ranks[] = $userRank->rankID;
				}
			}
		}
		if (MULTIRANK_ONLY_ONE_PER_GROUP) {
			/** @var UserRank[] $allowedGroupIDs */
			$allowedGroupIDs = [];
			foreach ($canUseRanks as $userRank) {
				$allowedGroupIDs[$userRank->groupID] = $userRank;
			}
			
			/** @var UserRank[] $groupIDs */
			$groupIDs = [];
			foreach ($ranks as $rankID) {
				if (isset($allRanks[$rankID])) {
					$newRank = $allRanks[$rankID];
					if (!isset($groupIDs[$newRank->groupID])) {
						$groupIDs[$newRank->groupID] = $newRank;
					} else {
						$lastRank = $groupIDs[$newRank->groupID];
						if ($newRank->requiredPoints > $lastRank->requiredPoints) {
							$groupIDs[$newRank->groupID] = $newRank;
						}
					}
				}
			}
			$ranks = [];
			foreach ($groupIDs as $groupID => $rank) {
				$ranks[] = $allowedGroupIDs[$groupID]->rankID;
			}
		}
		$ranks = array_unique($ranks);
		$sql = "UPDATE wcf" . WCF_N . "_user_option_value SET userOption" . $user->getUserOptionID("multirankIDs") . " = ? WHERE userID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([implode("\n", $ranks), $userID]);
	}
	
	/**
	 * Return for this user the list of used rank IDs
	 *
	 * @param User $user
	 *
	 * @return array
	 */
	private function getUserRank($user) {
		if ($user) {
			/** @noinspection PhpUndefinedFieldInspection */
			return explode("\n", $user->multirankIDs);
		}
		
		return [];
	}
}
