<?php

namespace wcf\data\multirank;

use wcf\data\DatabaseObjectEditor;
use wcf\data\IEditableCachedObject;
use wcf\system\cache\builder\MultirankCacheBuilder;
use wcf\system\cache\builder\MultirankNoParentCacheBuilder;
use wcf\system\WCF;

/**
 * @author    Olaf Braun
 * @copyright 2013-2019 Olaf Braun - Software Development
 * @license   WBB-Elite.de License <https://lizenz.wbb-elite.de/lizenz.html>
 * @package   de.wbb-elite.multirank
 * @category  Several rank graphics
 *
 * @method static Multirank        create(array $parameters = [])
 * @method        Multirank        getDecoratedObject()
 * @mixin        Multirank
 */
class MultirankEditor extends DatabaseObjectEditor implements IEditableCachedObject {
	
	/**
	 * @inheritdoc
	 */
	protected static $baseClass = Multirank::class;
	
	/**
	 * @inheritdoc
	 */
	public static function resetCache() {
		MultirankNoParentCacheBuilder::getInstance()->reset();
		MultirankCacheBuilder::getInstance()->reset();
	}
	
	/**
	 * Prepares the update of the position of this multirank object and return the new position id
	 *
	 * @param    integer $parentID
	 * @param    integer $position
	 *
	 * @return    integer
	 */
	public function updateposition($parentID, $position) {
		// correct invalid values
		if ($position === null) {
			$position = PHP_INT_MAX;
		}
		
		if ($parentID != $this->parentID) {
			$sql = "UPDATE	" . static::getDatabaseTableName() . "
				SET	position = position - 1
				WHERE	position > ?
					AND parentID = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([$this->position,
				$this->parentID,]);
			
			return static::getPosition($parentID, $position);
		} else {
			if ($position < $this->position) {
				$sql = "UPDATE	" . static::getDatabaseTableName() . "
					SET	position = position + 1
					WHERE	position >= ?
						AND position < ?
						AND parentID = ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute([$position,
					$this->position,
					$this->parentID,]);
			} else if ($position > $this->position) {
				$sql = "SELECT	MAX(position) AS position
					FROM	" . static::getDatabaseTableName() . "
					WHERE	parentID = ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute([$this->parentID]);
				$row = $statement->fetchArray();
				$maxposition = 0;
				if (!empty($row)) {
					$maxposition = intval($row['position']);
				}
				
				if ($position > $maxposition) {
					$position = $maxposition;
				}
				
				$sql = "UPDATE	" . static::getDatabaseTableName() . "
					SET	position = position - 1
					WHERE	position <= ?
						AND position > ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute([$position,
					$this->position]);
			}
			
			return $position;
		}
	}
	
	/**
	 * Returns the show order for a new category.
	 *
	 * @param    integer $parentCategoryID
	 * @param    integer $position
	 *
	 * @return    integer
	 */
	protected static function getPosition($parentCategoryID, $position) {
		// correct invalid values
		if ($position === null) {
			$position = PHP_INT_MAX;
		}
		
		$sql = "SELECT	MAX(position) AS position
			FROM	" . static::getDatabaseTableName() . "
			WHERE	parentID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([
			$parentCategoryID
		]);
		$row = $statement->fetchArray();
		$maxShowOrder = 0;
		if (!empty($row)) {
			$maxShowOrder = intval($row['position']);
		}
		
		if ($maxShowOrder && $position <= $maxShowOrder) {
			$sql = "UPDATE	" . static::getDatabaseTableName() . "
				SET	position = position + 1
				WHERE	position >= ?
					AND parentID = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([
				$position,
				$parentCategoryID
			]);
			
			return $position;
		}
		
		return $maxShowOrder + 1;
	}
	
}
