<?php

namespace wcf\data\multirank;

use wcf\data\DatabaseObject;
use wcf\data\user\rank\UserRank;
use wcf\data\user\rank\UserRankList;
use wcf\data\user\User;
use wcf\system\cache\builder\MultirankCacheBuilder;
use wcf\system\cache\builder\MultirankNoParentCacheBuilder;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\multirank\MultirankHandler;
use wcf\system\WCF;

/**
 * @author        Olaf Braun
 * @copyright     2013-2019 Olaf Braun - Software Development
 * @license       WBB-Elite.de License <https://lizenz.wbb-elite.de/lizenz.html>
 * @package       de.wbb-elite.multirank
 * @category      Several rank graphics
 *
 * @property integer $multirankID
 * @property integer $max
 * @property integer $position
 * @property integer $parentID
 * @property integer $objectTypeID
 * @property integer $type
 * @property string  $css
 * @property string  $title
 */
class Multirank extends DatabaseObject {
	
	/**
	 * @see    \wcf\data\DatabaseObject::$databaseTableName
	 */
	protected static $databaseTableName = 'multirank';
	
	/**
	 * @see    \wcf\data\DatabaseObject::$databaseTableIndexName
	 */
	protected static $databaseTableIndexName = 'multirankID';
	
	/**
	 * Returns a list of all multiranks wich have not a parent
	 *
	 * @return array
	 */
	public static function getMultiranks() {
		return MultirankNoParentCacheBuilder::getInstance()->getData();
	}
	
	/**
	 * Returns a list of all multiranks
	 *
	 * @return array
	 */
	public static function getMultiranksTemplate() {
		return MultirankCacheBuilder::getInstance()->getData();
	}
	
	/**
	 * Return all parent multirank objects
	 *
	 * @param integer $userID
	 *
	 * @return UserRank[]
	 */
	public static function getRanks($userID) {
		//reload user
		$user = new User((int)$userID);
		$groupIds = $user->getGroupIDs();
		$ranks = new UserRankList();
		if (MULTIRANK_ONLY_ONE_PER_GROUP) {
			$rankIDs = [];
			$conditionBuilder = new PreparedStatementConditionBuilder();
			if (count($groupIds) > 0) {
				$conditionBuilder->add('user_rank.groupID IN (?)', [$groupIds]);
			}
			if ($user->gender) {
				$conditionBuilder->add('user_rank.requiredGender IN (?)', [
					[
						0,
						$user->gender
					]
				]);
			} else {
				$conditionBuilder->add('user_rank.requiredGender = ?', [0]);
			}
			$sql = "SELECT user_rank.rankID
FROM wcf" . WCF_N . "_user_rank user_rank
LEFT JOIN wcf" . WCF_N . "_user_group user_group
ON(user_group.groupID = user_rank.groupID)
" . $conditionBuilder . "
AND user_rank.requiredPoints IN
(
	SELECT MAX(requiredPoints)
	FROM wcf" . WCF_N . "_user_rank
	WHERE groupID = user_group.groupID AND
	requiredPoints <= ?
	GROUP BY groupID
)
ORDER BY user_group.priority DESC, user_rank.requiredPoints DESC, user_rank.requiredGender DESC";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute(array_merge($conditionBuilder->getParameters(), [$user->activityPoints]));
			while ($row = $statement->fetchArray()) {
				$rankIDs[] = $row["rankID"];
			}
			$ranks->setObjectIDs($rankIDs);
		} else {
			if (count($groupIds) > 0) {
				$ranks->getConditionBuilder()->add('user_rank.groupID IN (?)', [$groupIds]);
			}
			$ranks->getConditionBuilder()->add('user_rank.requiredPoints <= ?', [$user->activityPoints]);
			if ($user->gender) {
				$ranks->getConditionBuilder()->add('user_rank.requiredGender IN (?)', [
					[
						0,
						$user->gender
					]
				]);
			} else {
				$ranks->getConditionBuilder()->add('user_rank.requiredGender = ?', [0]);
			}
			$ranks->sqlJoins = "LEFT JOIN	wcf" . WCF_N . "_user_group user_group
				ON		(user_group.groupID = user_rank.groupID)";
			$ranks->sqlOrderBy = "user_group.priority DESC, user_rank.requiredPoints DESC, user_rank.requiredGender DESC";
		}
		
		$ranks->readObjects();
		
		return $ranks->getObjects();
	}
	
	/**
	 * Returns a multirank to user object
	 *
	 * @return string
	 */
	public function getType() {
		$type = 'de.wbb-elite.multirank.acp.type.';
		switch ($this->type) {
			case 0:
				$type .= 'befor';
				break;
			case 1:
				$type .= 'insert';
				break;
			case 2:
				$type .= 'after';
				break;
		}
		
		return $type;
	}
	
	/**
	 * Returns the object type for this multirank setting
	 *
	 * @return \wcf\data\object\type\ObjectType
	 */
	public function getObjectType() {
		return MultirankHandler::getInstance()->getObjectType($this->objectTypeID)->getProcessor();
	}
	
	/**
	 * Return all parent multirank objects
	 *
	 * @return array
	 */
	public function getParentMultiranks() {
		$objects = [];
		$array = MultirankCacheBuilder::getInstance()->getData();
		foreach ($array as $object) {
			if ($object->parentID == $this->multirankID) {
				$objects[$object->multirankID] = $object;
			}
		}
		
		return $objects;
	}
	
	/**
	 * Returns the ID for a link
	 *
	 * @return string
	 */
	public function getIdentifier() {
		return 'multirankID' . $this->multirankID;
	}
}
