<?php

namespace wcf\acp\form;

use wcf\data\multirank\Multirank;
use wcf\data\multirank\MultirankAction;
use wcf\data\multirank\MultirankEditor;
use wcf\data\package\PackageCache;
use wcf\form\AbstractForm;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\UserInputException;
use wcf\system\language\I18nHandler;
use wcf\system\multirank\MultirankHandler;
use wcf\system\WCF;
use wcf\util\StringUtil;

/**
 * Multirank add form class
 *
 * @author        Olaf Braun
 * @copyright     2013-2019 Olaf Braun - Software Development
 * @license       WBB-Elite.de License <https://lizenz.wbb-elite.de/lizenz.html>
 * @package       de.wbb-elite.multirank
 * @category      Several rank graphics
 */
class MultirankAddForm extends AbstractForm {
	
	/**
	 * @see    \wcf\page\AbstractPage::$activeMenuItem
	 */
	public $activeMenuItem = 'wcf.acp.menu.link.multirank.add';
	/**
	 * @see    \wcf\page\AbstractPage::$neededPermissions
	 */
	public $neededPermissions = ['admin.multirank.canMange'];
	/**
	 * @see    \wcf\page\AbstractPage::$neededPermissions
	 */
	public $neededModules = ['MODULE_MULTIRANK'];
	/**
	 * Multirank title
	 *
	 * @var    string
	 */
	public $title = '';
	/**
	 * list of multirank object types
	 *
	 * @var    array<\wcf\data\object\type>
	 */
	public $objectTypes = [];
	/**
	 * active multirank object type id
	 *
	 * @var    integer
	 */
	public $objectTypeID = 0;
	/**
	 * active multirank object type
	 *
	 * @var    \wcf\data\object\type\ObjectType
	 */
	public $objectType = null;
	/**
	 * Multirank css value
	 *
	 * @var    string
	 */
	public $css = '';
	/**
	 * Type of append system
	 *
	 * @var    integer
	 */
	public $type = 0;
	/**
	 * Max ranks of this object, 0 = endlessly
	 *
	 * @var    integer
	 */
	public $max = 0;
	/**
	 * Position id of this object
	 *
	 * @var    integer
	 */
	public $position = 0;
	/**
	 * Parent id of this object
	 *
	 * @var    integer
	 */
	public $parentID = 0;
	/**
	 * Parent object list
	 *
	 * @var    array
	 */
	public $parents = [];
	
	/**
	 * @inheritdoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		I18nHandler::getInstance()->assignVariables();
		WCF::getTPL()->assign(['action'       => 'add',
		                       'title'        => $this->title,
		                       'css'          => $this->css,
		                       'position'     => $this->position,
		                       'parentID'     => $this->parentID,
		                       'max'          => $this->max,
		                       'parents'      => $this->parents,
		                       'objectType'   => $this->objectType,
		                       'objectTypeID' => $this->objectTypeID,
		                       'objectTypes'  => $this->objectTypes,
		                       'type'         => $this->type]);
	}
	
	/**
	 * @inheritdoc
	 */
	public function readFormParameters() {
		parent::readFormParameters();
		// read i18n values
		I18nHandler::getInstance()->readValues();
		// handle i18n plain input
		if (I18nHandler::getInstance()->isPlainValue('title')) $this->title = I18nHandler::getInstance()->getValue('title');
		// handle default input
		
		if (isset ($_POST['position'])) $this->position = intval($_POST['position']);
		if (isset ($_POST['parentID'])) $this->parentID = intval($_POST['parentID']);
		if (isset ($_POST['type'])) $this->type = intval($_POST['type']);
		if (isset ($_POST['max'])) $this->max = intval($_POST['max']);
		if (isset ($_POST['objectTypeID'])) $this->objectTypeID = intval($_POST['objectTypeID']);
		if (isset ($_POST['css'])) $this->css = StringUtil::trim($_POST['css']);
	}
	
	/**
	 * @inheritdoc
	 */
	public function readParameters() {
		parent::readParameters();
		$this->parents = Multirank::getMultiranks();
		$this->objectTypes = MultirankHandler::getInstance()->getMultriankTypes();
		if (count($this->objectTypes) == 0) {
			throw new IllegalLinkException();
		}
		I18nHandler::getInstance()->register('title');
	}
	
	/**
	 * @inheritdoc
	 */
	public function save() {
		parent::save();
		$this->objectAction = new MultirankAction([], 'create', ['data' => ['title'        => $this->title,
		                                                                    'position'     => $this->position,
		                                                                    'parentID'     => $this->parentID,
		                                                                    'max'          => $this->max,
		                                                                    'objectTypeID' => $this->objectTypeID,
		                                                                    'css'          => $this->css,
		                                                                    'type'         => $this->type]]);
		$returnValues = $this->objectAction->executeAction();
		// save i18n values
		$this->saveI18nValue($returnValues['returnValues'], 'title');
		$this->saved();
		I18nHandler::getInstance()->reset();
		$this->title = $this->search = $this->append = $this->dataUserID = $this->userID = '';
		$this->type = 0;
		WCF::getTPL()->assign(['success' => true]);
	}
	
	/**
	 * @inheritdoc
	 */
	public function validate() {
		parent::validate();
		// validate title
		if (!I18nHandler::getInstance()->validateValue('title')) {
			throw new UserInputException('title');
		}
		if (empty($this->objectTypeID) || $this->objectTypeID == 0) {
			throw new UserInputException('objectTypeID');
		}
		if (MultirankHandler::getInstance()->getObjectType($this->objectTypeID) == null) {
			throw new UserInputException('objectTypeID');
		}
		if ($this->max < 0) {
			throw new UserInputException('max');
		}
		if (!in_array($this->type, [0,
			1,
			2])
		) {
			throw new UserInputException('type');
		}
		if ($this->parentID != 0) {
			$multirank = new Multirank($this->parentID);
			if ($multirank === null || $multirank->multirankID == 0) {
				throw new UserInputException('parentID', 'notValid');
			}
			if (intval($multirank->parentID) != 0) {
				throw new UserInputException('parentID', 'notValid');
			}
		}
		
	}
	
	/**
	 * Saves i18n values.
	 *
	 * @param    \wcf\data\multirank\Multirank $object
	 * @param    string                        $columnName
	 */
	public function saveI18nValue(Multirank $object, $columnName) {
		if (!I18nHandler::getInstance()->isPlainValue($columnName)) {
			I18nHandler::getInstance()->save($columnName, 'de.wbb-elite.multirank' . $object->multirankID . '.title', 'de.wbb-elite.multirank', PackageCache::getInstance()->getPackageID('com.woltlab.wcf'));
			// update description
			$editor = new MultirankEditor($object);
			$editor->update([$columnName => 'de.wbb-elite.multirank' . $object->multirankID . '.title']);
		}
	}
}
