/**
 * @author    Olaf Braun
 * @copyright 2013-2019 Olaf Braun - Software Development
 * @license   WBB-Elite.de License <https://lizenz.wbb-elite.de/lizenz.html>
 * @package   de.wbb-elite.multirank
 * @category  Several rank graphics
 */
define(['Ajax', 'Dictionary', 'Dom/Util', 'Dom/ChangeListener'], function (Ajax, Dictionary, DomUtil, DomChangeListener) {
    "use strict";

    /**
     * @param identifier
     * @param container
     * @param userElement
     * @param dataUserId
     * @param appendContainer
     * @param type
     * @constructor
     */
    function MultiRankManager(identifier, container, userElement, dataUserId, appendContainer, type) {
        this.init(identifier, container, userElement, dataUserId, appendContainer, type);
    }

    MultiRankManager.prototype = {
        /**
         * Init the multirank manager
         * @param identifier
         * @param container
         * @param userElement
         * @param dataUserId
         * @param appendContainer
         * @param type
         * @private
         */
        init: function (identifier, container, userElement, dataUserId, appendContainer, type) {
            this._identifier = identifier;
            this._container = container;
            this._userElement = userElement;
            this._dataUserId = dataUserId;
            this._appendContainer = appendContainer;
            this._type = type;
            this._cache = new Dictionary();
            this._ajaxSend = false;
            this.load();
            DomChangeListener.add("multiRankChangeListener", this.load.bind(this));
        },
        /**
         * Find the position and load rang images
         * @private
         */
        load: function () {
            var objectIds = [];
            var self = this;
            this._getContainers(function (element) {
                var away = true;
                var findElement = elBySel(self._appendContainer, element);
                if (!findElement) return;

                switch (self._type) {
                    case 0:
                        findElement = elBySel('.multirankContainer', findElement.parentNode);
                        if (!findElement || findElement.length == 0) {
                            away = false;
                        }
                        break;
                    case 1:
                        findElement = elBySel('.multirankContainer', findElement);
                        if (!findElement || findElement.length == 0) {
                            away = false;
                        }
                        break;
                    case 2:
                        findElement = elBySel('.multirankContainer', findElement.parentNode);
                        if (!findElement || findElement.length == 0) {
                            away = false;
                        }
                        break;
                }
                if (!away) {
                    var userLink = '';
                    if (self._userElement == '') {
                        userLink = element;
                    } else {
                        userLink = elBySel(self._userElement, element);
                    }
                    if (!userLink) return;

                    if (elData(userLink, self._dataUserId) > 0 && !self._cache.has(elData(userLink, self._dataUserId))) {
                        objectIds.push(elData(userLink, self._dataUserId));
                    }
                }
            });
            if (objectIds.length > 0 && !self._ajaxSend) {
                self._ajaxSend = true;
                Ajax.apiOnce({
                    data: {
                        silent: true,
                        actionName: 'getMultirank',
                        className: 'wcf\\data\\multirank\\MultirankAction',
                        objectIDs: objectIds,
                        parameters: {
                            multirankID: self._identifier
                        }
                    }, success: function (data) {
                        self._ajaxSend = false;
                        if (data.returnValues.settings && data.returnValues.settings.length > 0) {
                            self._settings = data.returnValues.settings;
                        }
                        self._addToCache(data.returnValues.objects);
                    }
                });
            }
        },
        /**
         * Add items to cache
         *
         * @param {Object} data
         * @private
         */
        _addToCache: function (data) {
            var self = this;
            Object.keys(data).forEach(function (key) {
                self._cache.set(parseInt(key), data[key]);
            });
            this._addMultiranks();
        },
        /**
         * @param {Function} callback
         * @private
         */
        _getContainers: function (callback) {
            var containers = elBySelAll(this._container);
            if (containers && containers.length > 0) {
                forEach(containers, function (element) {
                    callback(element);
                });
            }
        },
        /**
         * Append the multi rank images
         * @private
         */
        _addMultiranks: function () {
            var self = this;
            this._getContainers(function (element) {
                var userLink = '';
                if (self._userElement == '') {
                    userLink = element;
                } else {
                    userLink = elBySel(self._userElement, element);
                }
                if (!userLink) return;
                var userID = elData(userLink, self._dataUserId);
                if (self._cache.has(userID)) {
                    var object = self._cache.get(userID);
                    var append = elCreate('div');
                    append.classList.add('multirankContainer');
                    if (self._settings && self._settings.length > 0) {
                        append.style.cssText = self._settings;
                    }
                    for (var i = 0; i < object.length; i++) {
                        append.appendChild(DomUtil.createFragmentFromHtml(object));
                    }
                    var findElement = elBySel(self._appendContainer, element);
                    if (!findElement) return;

                    switch (self._type) {
                        case 0:
                            var search = elBySel('.multirankContainer', findElement.parentNode);
                            if (!search || search.length == 0) {
                                findElement.parentNode.insertBefore(append, findElement);
                            }
                            break;
                        case 1:
                            search = elBySel('.multirankContainer', findElement);
                            if (!search || search.length == 0) {
                                findElement.appendChild(append);
                            }
                            break;
                        case 2:
                            search = elBySel('.multirankContainer', findElement.parentNode);
                            if (!search || search.length == 0) {
                                DomUtil.insertAfter(append, findElement);
                            }
                            break;
                    }
                }
            });
        }
    };
    return MultiRankManager;
});