<?php
namespace wcf\acp\form;
use wcf\data\user\group\UserGroupList;
use wcf\data\user\group\UserGroup;
use wcf\data\user\group\UserGroupEditor;
use wcf\form\AbstractForm;
use wcf\system\exception\UserInputException;
use wcf\system\WCF;

/**
 * Copies the user group rights from one user group to another
 * 
 * @author		Sonnenspeer
 * @copyright	2013 Sonnenspeer
 * @license		CC BY-ND 3.0 DE <http://creativecommons.org/licenses/by-nd/3.0/de/legalcode>
 * @package		de.sonnenspeer.wcf.acp.grouprights.copy
 * @subpackage	acp.page
 */
class GroupRightsCopyForm extends AbstractForm  {
	/**
	 * @see	\wcf\page\AbstractPage::$activeMenuItem
	 */
	public $templateName = 'groupRightsCopy';
	
	/**
	 * @see	\wcf\page\AbstractPage::$activeMenuItem
	 */
	public $activeMenuItem = 'wcf.acp.menu.link.group.rights.copy';
	
	/**
	 * @see	\wcf\page\AbstractPage::$neededPermissions
	 */
	public $neededPermissions = array('admin.user.canEditGroup', 'admin.user.canCreateGroup');
	
	/**
	 * Source
	 */
	private $sourceGroupID = null;
	
	/**
	 * Target
	 */
	private $targetGroupIDs = array();
	
	/**
	 * @see	\wcf\page\IPage::readParameters()
	 */
	public function readParameters() {
		parent::readParameters();
		
		if (isset($_REQUEST['sourceGroupID'])) $this->sourceGroupID = intval($_REQUEST['sourceGroupID']);
		if (isset($_REQUEST['targetGroupIDs'])) $this->targetGroupIDs = (empty($_REQUEST['targetGroupIDs'])) ? array() : $_REQUEST['targetGroupIDs'];
	}
	
	/**
	 * @see    \wcf\form\IForm::validate()
	 */
	public function validate() {
		parent::validate();
		
		$this->validateSourceGroupID();
		
		$this->validateTargetGroupIDs();
	}
	
	/**
	 * Validate if source is empty nor
	 */
	protected function validateSourceGroupID() {		
		if ($this->sourceGroupID == null) {			
			throw new UserInputException('sourceGroupID', 'noSelection');
		}
	}
	
	/**
	 * Validate if target is empty nor
	 */
	protected function validateTargetGroupIDs() {
		if (count($this->targetGroupIDs) == 0) {
			throw new UserInputException('targetGroupIDs', 'noSelection');
		}
	}
	
	/**
	 * @see	\wcf\form\IForm::save()
	 */
	public function save() {
		parent::save();
		
		$options = array();
		$objects = array();
		
		// source group permissions (read by sql)
		$sql = "SELECT optionID, optionValue 
				FROM wcf".WCF_N."_user_group_option_value
				WHERE groupID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute(array($this->sourceGroupID));
		
		while($row = $statement->fetchArray()) {
			$options[$row['optionID']] = $row['optionValue'];
		}
		
		// get editor objects of target group ids
		foreach($this->targetGroupIDs AS $groupID) {
			$group = new UserGroup($groupID);
			$objects[$groupID] = new UserGroupEditor($group);
		} 
		
		// overwrite options for each target group
		foreach ($objects as $object) {
			$object->updateGroupOptions($options);
		}
		
		$this->saved();
		
		// show success
		WCF::getTPL()->assign(array(
			'success' => true
		));	
	}
	
	/**
	 * @see	\wcf\page\IPage::assignVariables()
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		$groupList = new UserGroupList();
		$groupList->readObjects();
		
		WCF::getTPL()->assign(array(
			'groupList' => $groupList->getObjects(),
			'sourceGroupID' => $this->sourceGroupID
		));
	}
}
