<?php
namespace wcf\system\option\user\group;
use wcf\data\user\group\UserGroup;
use wcf\data\user\User;
use wcf\system\cache\builder\UserGroupPermissionCacheBuilder;
use wcf\data\user\group\option\EffectiveUserGroupOption;

/**
 * OptionHandler implementation to show the effective user group rights
 * of a certain user.
 *
 * @author	Fabian Gossner
 * @copyright 	2018 Fabian Gossner
 * @license	Creative Commons Attribution-NonCommercial-ShareAlike <http://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>
 */
class EffectiveUserGroupOptionHandler extends UserGroupOptionHandler {
	/**
	 * user that is being looked up
	 * @var	User
	 */
	protected $user = null;
	
	/**
	 * user group data and permissions
	 * @var	mixed[][]
	 */
	protected $userGroupData = null;
	
	/**
	 *
	 * @var array
	 */
	protected $allGroupData = [];
	
	/**
	 * Sets the effective user group option values for the given user.
	 *
	 * @param	User	$user
	 */
	public function setUser(User $user) {
		$this->userGroupData = null;
		$this->allGroupData = [];
		$this->optionValues = [];
		$this->user = $user;
		
		$this->init();
		foreach ($this->options as $option) {
			$this->optionValues[$option->optionName] = $this->getPermission($option->optionName);
		}
		
		$this->loadPermissionsForAllGroups();
	}
	
	/**
	 * @inheritDoc
	 */
	public function getOption($optionName) {
		$optionData = parent::getOption($optionName);
		
		if (isset($this->optionValues[$optionName])) {
			/** @noinspection PhpUndefinedMethodInspection */
			$optionData['object'] = new EffectiveUserGroupOption($optionData['object']);
			$optionData['object']->setOptionValue($this->optionValues[$optionName]);
			
			// fetch groups which have the effective permission value as their option value
			foreach ($this->allGroupData as $groupData) {
				if (isset($groupData['permissions'][$optionName]) && $groupData['permissions'][$optionName] == $this->optionValues[$optionName]) {
					$optionData['object']->addUserGroupNameToList($groupData['groupTitle']);
				}
			}
		}
		
		return $optionData;
	}
	
	/**
	 * Loads group data for a certain user from cache.
	 *
	 * @see	SessionHandler::loadGroupData()
	 */
	protected function loadGroupData() {
		if ($this->userGroupData !== null) return;
		
		$groupIDs = $this->user->getGroupIDs();
		
		// get group data from cache
		$this->userGroupData = UserGroupPermissionCacheBuilder::getInstance()->getData($groupIDs);
		if (isset($this->userGroupData['groupIDs']) && $this->userGroupData['groupIDs'] != $groupIDs) {
			$this->userGroupData = [];
		}
	}
	
	/**
	 * Returns the value of the permission with the given name.
	 *
	 * @see	SessionHandler::getPermission()
	 *
	 * @param	string		$permission
	 * @return	mixed		permission value
	 */
	protected function getPermission($permission) {
		// check if a users only permission is checked for a guest and return
		// false if that is the case
		if (!$this->user->userID && in_array($permission, $this->usersOnlyPermissions)) {
			return false;
		}
		
		$this->loadGroupData();
		
		if (!isset($this->userGroupData[$permission])) return false;
		return $this->userGroupData[$permission];
	}
	
	/**
	 * Loads the group data for each group where the user is a member.
	 */
	protected function loadPermissionsForAllGroups() {
		/** @var UserGroup $group */
		foreach (UserGroup::getGroupsByIDs($this->user->getGroupIDs()) as $group) {
			$this->allGroupData[$group->groupID] = [
				'permissions' => UserGroupPermissionCacheBuilder::getInstance()->getData([$group->groupID]),
				'groupTitle' => $group->getTitle()
			];
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function setUserGroup(UserGroup $group) {
		throw new \LogicException("Setting a user group is not supported, as this implementation only works with an user object.");
	}
}
