<?php
namespace wcf\data\user\group\option;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\user\group\UserGroup;
use wcf\system\bbcode\SimpleMessageParser;
use wcf\system\WCF;
use wcf\util\DateUtil;
use wcf\util\FileUtil;
use wcf\util\OptionUtil;
use wcf\util\StringUtil;

/**
 * Parses user group options into a readonly representation.
 *
 * @author	Fabian Gossner
 * @copyright 	2018 Fabian Gossner
 * @license	Creative Commons Attribution-NonCommercial-ShareAlike <http://creativecommons.org/licenses/by-nc-sa/4.0/legalcode>
 */
class EffectiveUserGroupOption extends DatabaseObjectDecorator {
	/**
	 * @inheritDoc
	 */
	protected static $baseClass = UserGroupOption::class;
	
	/**
	 * option value
	 * @var	string
	 */
	protected $optionValue = '';
	
	/**
	 * list of user group names which have the effective permission
	 * value as their option value
	 * @var string[]
	 */
	protected $inGroups = [];
	
	/**
	 * Returns the value of this option.
	 *
	 * @return	string
	 */
	public function getOptionValue() {
		return $this->optionValue;
	}
	
	/**
	 * Sets the value of this option.
	 *
	 * @param	string		$value
	 */
	public function setOptionValue($value) {
		$this->optionValue = $value;
	}
	
	/**
	 * Adds the given user group name to the list of user group names which
	 * have the effective permission value as their option value.
	 *
	 * @param	string		$name
	 */
	public function addUserGroupNameToList($name) {
		$this->inGroups[] = $name;
	}
	
	/**
	 * Returns the user group name list in a human readable way.
	 *
	 * @return string
	 */
	public function getFormattedUserGroupNameList() {
		return implode(', ', $this->inGroups);
	}
	
	/**
	 * Returns the formatted value of this user group option.
	 *
	 * @return	string
	 */
	public function getFormattedOptionValue() {
		switch ($this->optionType) {
			case 'boolean':
				return WCF::getLanguage()->get('wcf.acp.option.type.boolean.'.($this->optionValue ? 'yes' : 'no'));
			
			case 'date':
				$year = $month = $day = 0;
				$optionValue = explode('-', $this->optionValue);
				if (isset($optionValue[0])) $year = intval($optionValue[0]);
				if (isset($optionValue[1])) $month = intval($optionValue[1]);
				if (isset($optionValue[2])) $day = intval($optionValue[2]);
				return DateUtil::format(DateUtil::getDateTimeByTimestamp(gmmktime(12, 1, 1, $month, $day, $year)), DateUtil::DATE_FORMAT);
			
			case 'float':
				return StringUtil::formatDouble(intval($this->optionValue));
			
			case 'integer':
			case 'inverseInteger':
			case 'infiniteInteger':
			case 'infiniteInverseInteger':
				$value = StringUtil::formatInteger(intval($this->optionValue));
				if ($this->suffix) $value .= ' ' . WCF::getLanguage()->get('wcf.acp.option.suffix.'.$this->suffix);
				return $value;
			
			case 'radioButton':
			case 'select':
				$selectOptions = OptionUtil::parseSelectOptions($this->selectOptions);
				if (isset($selectOptions[$this->optionValue])) return WCF::getLanguage()->get($selectOptions[$this->optionValue]);
				return '';
			
			case 'multiSelect':
			case 'checkboxes':
				$selectOptions = OptionUtil::parseSelectOptions($this->selectOptions);
				$values = explode("\n", $this->optionValue);
				$result = '';
				foreach ($values as $value) {
					if (isset($selectOptions[$value])) {
						if (!empty($result)) $result .= "<br>\n";
						$result .= WCF::getLanguage()->get($selectOptions[$value]);
					}
				}
				return $result;
			
			case 'textarea':
				return SimpleMessageParser::getInstance()->parse($this->optionValue);
			
			case 'fileSize':
				return FileUtil::formatFilesize($this->optionValue);
				
			case 'URL':
				return StringUtil::getAnchorTag($this->optionValue);
				
			case 'BBCodeSelect':
				$values = explode(",", $this->optionValue);
				$result = '';
				foreach ($values as $value) {
					if (!empty($result)) $result .= "<br>\n";
					$result .= $value;
				}
				return $result;
				
			case 'userGroups':
				// get selected groups
				$selectedGroups = explode(',', $this->optionValue);
				
				// get all groups
				$groups = UserGroup::getGroupsByType();
				
				$result = '';
				foreach ($groups as $group) {
					if (in_array($group->groupID, $selectedGroups)) {
						if (!empty($result)) $result .= "<br>\n";
						$result .= $group->getName();
					}
				}
				return $result;
			
			default:
				$value = StringUtil::encodeHTML($this->optionValue);
				if ($this->suffix) $value .= ' ' . WCF::getLanguage()->get('wcf.acp.option.suffix.'.$this->suffix);
				
				return $value;
		}
	}
}