<?php
namespace wcf\system\user\notification\event;
use wcf\data\user\group\application\UserGroupApplication;
use wcf\data\user\group\UserGroup;
use wcf\data\user\notification\UserNotification;
use wcf\data\user\User;
use wcf\data\user\UserProfile;
use wcf\system\user\notification\object\IUserNotificationObject;
use wcf\system\request\LinkHandler;
use wcf\system\user\notification\object\UserGroupApplicationUserNotificationObject;

/**
 * Notification for the applicant when their application is edited.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\System\User\Notification\Event
 *
 * @method	UserGroupApplicationUserNotificationObject	getUserNotificationObject()
 */
class UserGroupApplicationEditedUserNotificationEvent extends AbstractUserNotificationEvent implements ITestableUserNotificationEvent {
	use TTestableUserGroupApplicationRelatedUserNotificationEvent;
	use TTestableUserNotificationEvent;
	
	/**
	 * @inheritDoc
	 */
	public function setObject(UserNotification $notification, IUserNotificationObject $object, UserProfile $author, array $additionalData = []) {
		parent::setObject($notification, $object, $author, $additionalData);
		
		// if the notification is processed by a group leader, the author
		// of the application can no longer be used as the author of the notification
		// which is instead given by authorID and authorName
		if (isset($additionalData['authorID']) && isset($additionalData['authorName'])) {
			$author = UserProfile::getUserProfile($additionalData['authorID']);
			if ($author === null) {
				$author = new UserProfile(new User(null, [
					'username' => $additionalData['authorName']
				]));
			}
			
			$this->author = $author;
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getLink() {
		if ($this->getUserNotificationObject()->status == UserGroupApplication::STATUS_ACCEPTED || $this->getUserNotificationObject()->status == UserGroupApplication::STATUS_CLOSED) {
			return LinkHandler::getInstance()->getLink('UserGroupList');
		}
		else {
			return LinkHandler::getInstance()->getLink('UserGroupApplicationEdit', [
				'id' => $this->getUserNotificationObject()->applicationID
			]);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTitle() {
		return $this->getLanguage()->getDynamicVariable('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.edited.title', [
			'newStatus' => $this->additionalData['newStatus']
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getMessage() {
		$group = UserGroup::getGroupByID($this->additionalData['groupID']);
		if ($group === null) $group = new UserGroup(null, []);
		
		return $this->getLanguage()->getDynamicVariable('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.edited.message', [
			'application' => $this->userNotificationObject,
			'author' => $this->author,
			'group' => $group,
			'newStatus' => $this->additionalData['newStatus']
		]);
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getEmailMessage($notificationType = 'instant') {
		$group = UserGroup::getGroupByID($this->additionalData['groupID']);
		if ($group === null) $group = new UserGroup(null, []);
		
		return [
			'template' => 'email_notification_moderatedUserGroup_applicationEdited',
			'application' => 'wcf',
			'variables' => [
				'group' => $group,
				'newStatus' => $this->additionalData['newStatus'],
				'application' => new UserGroupApplication($this->getUserNotificationObject()->applicationID)
			]
		];
	}
	
	/**
	 * maps the ids of the created user group application to their status
	 * @var	string[]
	 */
	protected static $testStatusMap = [];
	
	/**
	 * @inheritDoc
	 */
	public static function getTestAdditionalData(IUserNotificationObject $object) {
		/** @var UserGroupApplicationUserNotificationObject $object */
		
		$status = self::$testStatusMap[$object->applicationID];
		
		return [
			'groupID' => $object->groupID,
			'newStatus' => $status,
			'authorID' => $object->applicant,
			'authorName' => $object->username
		];
	}
	
	/**
	 * @inheritDoc
	 */
	public static function getTestObjects(UserProfile $recipient, UserProfile $author) {
		$acceptedUserGroupApplication = new UserGroupApplicationUserNotificationObject(self::createTestUserGroupApplication($recipient, $author));
		$rejectedUserGroupApplication = new UserGroupApplicationUserNotificationObject(self::createTestUserGroupApplication($recipient, $author));
		$closedUserGroupApplication = new UserGroupApplicationUserNotificationObject(self::createTestUserGroupApplication($recipient, $author));
		
		self::$testStatusMap = [
			$acceptedUserGroupApplication->applicationID => 'accepted',
			$rejectedUserGroupApplication->applicationID => 'rejected',
			$closedUserGroupApplication->applicationID => 'closed'
		];
		
		return [
			$acceptedUserGroupApplication,
			$rejectedUserGroupApplication,
			$closedUserGroupApplication
		];
	}
}
