<?php
namespace wcf\system\user\notification\event;
use wcf\data\user\group\ModeratedUserGroup;
use wcf\data\user\group\UserGroup;
use wcf\data\user\UserProfile;
use wcf\system\request\LinkHandler;
use wcf\system\user\notification\object\IUserNotificationObject;
use wcf\system\user\notification\object\UserGroupApplicationUserNotificationObject;

/**
 * Notification for the group leaders when a new application is created.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\System\User\Notification\Event
 * 
 * @method	UserGroupApplicationUserNotificationObject	getUserNotificationObject()
 */
class UserGroupApplicationCreatedUserNotificationEvent extends AbstractUserNotificationEvent implements ITestableUserNotificationEvent {
	use TTestableUserGroupApplicationRelatedUserNotificationEvent;
	use TTestableUserNotificationEvent;
	
	/**
	 * @inheritDoc
	 */
	public function getLink() {
		return LinkHandler::getInstance()->getLink('UserGroupManageApplicationEdit', [
			'id' => $this->getUserNotificationObject()->applicationID
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTitle() {
		return $this->getLanguage()->get('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.created.title');
	}
	
	/**
	 * @inheritDoc
	 */
	public function getMessage() {
		$group = UserGroup::getGroupByID($this->additionalData['groupID']);
		if ($group === null) $group = new UserGroup(null, []);
		
		return $this->getLanguage()->getDynamicVariable('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.created.message', [
			'application' => $this->userNotificationObject,
			'author' => $this->author,
			'group' => $group
		]);
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getEmailMessage($notificationType = 'instant') {
		$group = UserGroup::getGroupByID($this->additionalData['groupID']);
		if ($group === null) $group = new UserGroup(null, []);
		
		return [
			'message-id' => 'com.woltlab.wcf.moderatedUserGroup.application/'.$this->getUserNotificationObject()->applicationID,
			'template' => 'email_notification_moderatedUserGroup_applicationCreated',
			'application' => 'wcf',
			'variables' => [
				'group' => $group,
				'application' => $this->getUserNotificationObject()
			]
		];
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function isVisible() {
		return ModeratedUserGroup::isGroupLeader();
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function checkAccess() {
		$group = UserGroup::getGroupByID($this->additionalData['groupID']);
		if ($group === null) return false;
		
		$moderatedUserGroup = new ModeratedUserGroup($group);
		
		return $moderatedUserGroup->isLeader();
	}
	
	/**
	 * maps the ids of the created user group application to their status
	 * @var	string[]
	 */
	protected static $statusMap = [];
	
	/**
	 * @inheritDoc
	 */
	public static function getTestAdditionalData(IUserNotificationObject $object) {
		/** @var UserGroupApplicationUserNotificationObject $object */
		
		return [
			'groupID' => $object->groupID
		];
	}
	
	/**
	 * @inheritDoc
	 */
	public static function getTestObjects(UserProfile $recipient, UserProfile $author) {
		return [new UserGroupApplicationUserNotificationObject(self::createTestUserGroupApplication($recipient, $author))];
	}
}
