<?php
namespace wcf\system\user\notification\event;
use wcf\data\user\group\application\UserGroupApplication;
use wcf\data\user\group\ModeratedUserGroup;
use wcf\data\user\notification\UserNotification;
use wcf\data\user\UserProfile;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\email\Email;
use wcf\system\request\LinkHandler;
use wcf\system\user\notification\object\CommentUserNotificationObject;
use wcf\system\user\notification\object\IUserNotificationObject;

/**
 * User notification event for user group application commments.
 * 
 * @author	Matthias Schmidt
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\System\User\Notification\Event
 * @since	3.0
 * 
 * @method	CommentUserNotificationObject	getUserNotificationObject()
 */
class UserGroupApplicationCommentUserNotificationEvent extends AbstractUserNotificationEvent implements ITestableUserNotificationEvent {
	use TTestableCommentUserNotificationEvent;
	use TTestableUserGroupApplicationCommentUserNotificationEvent {
		TTestableUserGroupApplicationCommentUserNotificationEvent::canBeTriggeredByGuests insteadof TTestableCommentUserNotificationEvent;
	}
	
	/**
	 * user group application object the notifications (indirectly) belong to
	 * @var	UserGroupApplication
	 */
	protected $userGroupApplication;
	
	/**
	 * @inheritDoc
	 */
	protected $stackable = true;
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function checkAccess() {
		// the application does not exists anymore (the group may have been deleted or the application has been revoked)
		if (!$this->userGroupApplication->applicationID) return false;
		
		return $this->userGroupApplication->getGroup()->isLeader();
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getEmailMessage($notificationType = 'instant') {
		$messageID = '<com.woltlab.wcf.moderatedUserGroup.application/'.$this->getUserNotificationObject()->objectID.'@'.Email::getHost().'>';
		
		return [
			'message-id' => 'com.woltlab.wcf.moderatedUserGroup.application.comment/'.$this->getUserNotificationObject()->commentID,
			'template' => 'email_notification_comment',
			'application' => 'wcf',
			'in-reply-to' => [$messageID],
			'references' => [$messageID],
			'variables' => [
				'commentID' => $this->getUserNotificationObject()->commentID,
				'userGroupApplication' => $this->userGroupApplication,
				'languageVariablePrefix' => 'wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.comment'
			]
		];
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getEventHash() {
		return sha1($this->eventID . '-' . $this->userGroupApplication->applicationID);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getLink() {
		return LinkHandler::getInstance()->getLink('UserGroupManageApplicationEdit', [
			'id' => $this->userGroupApplication->applicationID
		], '#comments');
	}
	
	/**
	 * @inheritDoc
	 */
	public function getMessage() {
		$authors = $this->getAuthors();
		if (count($authors) > 1) {
			if (isset($authors[0])) {
				unset($authors[0]);
			}
			$count = count($authors);
			
			return $this->getLanguage()->getDynamicVariable('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.comment.message.stacked', [
				'author' => $this->author,
				'authors' => array_values($authors),
				'count' => $count,
				'others' => $count - 1,
				'userGroupApplication' => $this->userGroupApplication
			]);
		}
		
		return $this->getLanguage()->getDynamicVariable('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.comment.message', [
			'author' => $this->author,
			'commentID' => $this->getUserNotificationObject()->commentID,
			'userGroupApplication' => $this->userGroupApplication
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTitle() {
		$count = count($this->getAuthors());
		if ($count > 1) {
			return $this->getLanguage()->getDynamicVariable('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.comment.title.stacked', [
				'count' => $count,
				'timesTriggered' => $this->notification->timesTriggered
			]);
		}
		
		return $this->getLanguage()->get('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.comment.title');
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function isVisible() {
		return ModeratedUserGroup::isGroupLeader();
	}
	
	/**
	 * @inheritDoc
	 */
	protected function prepare() {
		UserProfileRuntimeCache::getInstance()->cacheObjectID($this->userGroupApplication->applicant);
	}
	
	/**
	 * @inheritDoc
	 */
	public function setObject(UserNotification $notification, IUserNotificationObject $object, UserProfile $author, array $additionalData = []) {
		parent::setObject($notification, $object, $author, $additionalData);
		
		$this->userGroupApplication = new UserGroupApplication($this->getUserNotificationObject()->objectID);
	}
}
