<?php
namespace wcf\system\user\notification\event;
use wcf\data\user\group\application\UserGroupApplication;
use wcf\data\user\group\ModeratedUserGroup;
use wcf\data\user\UserProfile;
use wcf\system\cache\runtime\CommentRuntimeCache;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\email\Email;
use wcf\system\request\LinkHandler;
use wcf\system\user\notification\object\CommentResponseUserNotificationObject;

/**
 * User notification event for user group application commments.
 *
 * @author	Matthias Schmidt
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\System\User\Notification\Event
 * @since	3.0
 * 
 * @method	CommentResponseUserNotificationObject	getUserNotificationObject()
 */
class UserGroupApplicationCommentResponseUserNotificationEvent extends AbstractSharedUserNotificationEvent implements ITestableUserNotificationEvent {
	use TTestableCommentResponseUserNotificationEvent;
	use TTestableUserGroupApplicationCommentUserNotificationEvent {
		TTestableUserGroupApplicationCommentUserNotificationEvent::canBeTriggeredByGuests insteadof TTestableCommentResponseUserNotificationEvent;
	}
	
	/**
	 * @inheritDoc
	 */
	protected $stackable = true;
	
	/**
	 * user group application object the notifications (indirectly) belong to
	 * @var	UserGroupApplication
	 */
	protected $userGroupApplication;
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function checkAccess() {
		// the application does not exists anymore (the group may have been deleted or the application has been revoked)
		if (!$this->getUserGroupApplication()->applicationID) return false;
		
		return $this->getUserGroupApplication()->getGroup()->isLeader();
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getEmailMessage($notificationType = 'instant') {
		$comment = CommentRuntimeCache::getInstance()->getObject($this->getUserNotificationObject()->commentID);
		if ($comment->userID) {
			$commentAuthor = UserProfileRuntimeCache::getInstance()->getObject($comment->userID);
		}
		else {
			$commentAuthor = UserProfile::getGuestUserProfile($comment->username);
		}
		
		$messageID = '<com.woltlab.wcf.moderatedUserGroup.application.comment/'.$comment->commentID.'@'.Email::getHost().'>';
		
		return [
			'template' => 'email_notification_commentResponse',
			'application' => 'wcf',
			'in-reply-to' => [$messageID],
			'references' => [$messageID],
			'variables' => [
				'commentAuthor' => $commentAuthor,
				'commentID' => $this->getUserNotificationObject()->commentID,
				'userGroupApplication' => $this->getUserGroupApplication(),
				'responseID' => $this->getUserNotificationObject()->responseID,
				'languageVariablePrefix' => 'wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.commentResponse'
			]
		];
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getEventHash() {
		return sha1($this->eventID . '-' . $this->getUserNotificationObject()->commentID);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getLink() {
		return LinkHandler::getInstance()->getLink('UserGroupManageApplicationEdit', [
			'id' => $this->getUserGroupApplication()->applicationID
		], '#comment' . $this->getUserNotificationObject()->commentID);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getMessage() {
		$authors = $this->getAuthors();
		if (count($authors) > 1) {
			if (isset($authors[0])) {
				unset($authors[0]);
			}
			$count = count($authors);
			
			return $this->getLanguage()->getDynamicVariable('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.commentResponse.message.stacked', [
				'authors' => array_values($authors),
				'count' => $count,
				'commentID' => $this->getUserNotificationObject()->commentID,
				'others' => $count - 1,
				'userGroupApplication' => $this->getUserGroupApplication()
			]);
		}
		
		$comment = CommentRuntimeCache::getInstance()->getObject($this->getUserNotificationObject()->commentID);
		if ($comment->userID) {
			$commentAuthor = UserProfileRuntimeCache::getInstance()->getObject($comment->userID);
		}
		else {
			$commentAuthor = UserProfile::getGuestUserProfile($comment->username);
		}
		
		return $this->getLanguage()->getDynamicVariable('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.commentResponse.message', [
			'author' => $this->author,
			'commentAuthor' => $commentAuthor,
			'commentID' => $this->getUserNotificationObject()->commentID,
			'responseID' => $this->getUserNotificationObject()->responseID,
			'userGroupApplication' => $this->getUserGroupApplication()
		]);
	}
	
	/**
	 * Returns the user group application object the responded to comment belongs to.
	 *
	 * @return	UserGroupApplication
	 */
	public function getUserGroupApplication() {
		if ($this->userGroupApplication === null) {
			$comment = CommentRuntimeCache::getInstance()->getObject($this->getUserNotificationObject()->commentID);
			
			$this->userGroupApplication = new UserGroupApplication($comment->objectID);
			UserProfileRuntimeCache::getInstance()->cacheObjectID($this->userGroupApplication->applicant);
		}
		
		return $this->userGroupApplication;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTitle() {
		$count = count($this->getAuthors());
		if ($count > 1) {
			return $this->getLanguage()->getDynamicVariable('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.commentResponse.title.stacked', [
				'count' => $count,
				'timesTriggered' => $this->notification->timesTriggered
			]);
		}
		
		return $this->getLanguage()->get('wcf.user.notification.com.woltlab.wcf.moderatedUserGroup.application.commentResponse.title');
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function isVisible() {
		return ModeratedUserGroup::isGroupLeader();
	}
	
	/**
	 * @inheritDoc
	 */
	protected function prepare() {
		CommentRuntimeCache::getInstance()->cacheObjectID($this->getUserNotificationObject()->commentID);
		UserProfileRuntimeCache::getInstance()->cacheObjectID($this->additionalData['userID']);
	}
}
