<?php
namespace wcf\system\event\listener;
use wcf\acp\form\UserGroupAddForm;
use wcf\acp\form\UserGroupEditForm;
use wcf\data\user\group\UserGroup;
use wcf\data\user\UserList;
use wcf\system\exception\UserInputException;
use wcf\system\WCF;
use wcf\util\ArrayUtil;

/**
 * Handles groupType and group leaders.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\System\Event\Listener
 */
class UserGroupAddFormModeratedGroupListener implements IParameterizedEventListener {
	/**
	 * instance of UserGroupAddForm
	 * @var	UserGroupAddForm|UserGroupEditForm
	 */
	protected $eventObj;
	
	/**
	 * group type
	 * @var	integer
	 */
	protected $groupType = 4;
	
	/**
	 * array of leader given by the user
	 * @var	string[]
	 */
	protected $leader = [];
	
	/**
	 * userlist containing the group leaders
	 * @var	UserList
	 */
	protected $leaderList;
	
	/**
	 * is the eventlistener disabled for this group
	 * @var	boolean
	 */
	private $disable = false;
	
	/**
	 * @inheritDoc
	 */
	public function execute($eventObj, $className, $eventName, array &$parameters) {
		$this->eventObj = $eventObj;
		
		if ($this->eventObj instanceof UserGroupEditForm && is_object($this->eventObj->group)) {
			switch ($this->eventObj->group->groupType) {
				// default groups must have the default groupType
				case UserGroup::EVERYONE:
				case UserGroup::GUESTS:
				case UserGroup::USERS:
					$this->disable = true;
				break;
			}
		}
		
		if ($this->disable) return;
		$this->$eventName();
	}
	
	/**
	 * Handles the assignVariables event.
	 */
	protected function assignVariables() {
		WCF::getTPL()->assign([
			'groupType' => $this->groupType,
			'leader' => implode(', ', $this->leader)
		]);
	}
	
	/**
	 * Handles the readData event.
	 * This is only called in UserGroupEditForm.
	 */
	protected function readData() {
		if (empty($_POST)) {
			$this->groupType = $this->eventObj->group->groupType;
			
			$this->leaderList = new UserList();
			$this->leaderList->getConditionBuilder()->add('user_table.userID IN (
				SELECT	leaderID
				FROM	wcf'.WCF_N.'_user_group_leader
				WHERE	groupID = ?
			)', [$this->eventObj->group->groupID]);
			$this->leaderList->sqlOrderBy = 'user_table.username ASC';
			$this->leaderList->readObjects();
			
			$this->leader = [];
			foreach ($this->leaderList as $user) {
				$this->leader[] = $user->username;
			}
		}
	}
	
	/**
	 * Handles the readFormParameters event.
	 */
	protected function readFormParameters() {
		if (isset($_POST['groupType'])) $this->groupType = intval($_POST['groupType']);
		if (isset($_POST['leader'])) {
			$leaders = ArrayUtil::trim(explode(',', $_POST['leader']));
			
			// remove duplicates
			foreach ($leaders as $leader) {
				$this->leader[mb_strtolower($leader)] = $leader;
			}
		}
	}
	
	/**
	 * Handles the save event.
	 */
	protected function save() {
		if ($this->eventObj instanceof UserGroupEditForm) {
			$this->eventObj->additionalFields = array_merge($this->eventObj->additionalFields, [
				'groupType' => $this->groupType
			]);
			
			// save group leaders
			$sql = "DELETE FROM wcf".WCF_N."_user_group_leader WHERE groupID = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([$this->eventObj->group->groupID]);
			
			if ($this->leaderList !== null) {
				$sql = "INSERT INTO wcf".WCF_N."_user_group_leader (groupID, leaderID) VALUES (?, ?)";
				$statement = WCF::getDB()->prepareStatement($sql);
				foreach ($this->leaderList as $user) {
					$statement->execute([$this->eventObj->group->groupID, $user->userID]);
				}
			}
			
			// delete application when groupType is changed to a type that does not support them
			// moderated, closed moderated
			if ($this->groupType != 6 && $this->groupType != 7) {
				$sql = "DELETE FROM wcf".WCF_N."_user_group_application WHERE groupID = ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute([$this->eventObj->group->groupID]);
			}
		}
		else {
			$this->eventObj->additionalFields = array_merge($this->eventObj->additionalFields, [
				'groupType' => $this->groupType
			]);
		}
	}
	
	/**
	 * Handles the saved event.
	 * This is only called in UserGroupAddForm.
	 */
	protected function saved() {
		$returnValues = $this->eventObj->objectAction->getReturnValues();
		$groupID = $returnValues['returnValues']->groupID;
		
		if ($this->leaderList !== null) {
			// save group leaders
			$sql = "INSERT INTO wcf".WCF_N."_user_group_leader (groupID, leaderID) VALUES (?, ?)";
			$statement = WCF::getDB()->prepareStatement($sql);
			foreach ($this->leaderList as $user) {
				$statement->execute([$groupID, $user->userID]);
			}
		}
		
		$this->leader = [];
	}
	
	/**
	 * Handles the validate event.
	 */
	protected function validate() {
		try {
			if ($this->groupType > 7 || $this->groupType < 4) {
				throw new UserInputException('groupType', 'invalid');
			}
		}
		catch (UserInputException $e) {
			$this->eventObj->errorType[$e->getField()] = $e->getType();
		}
		
		try {
			if (!empty($this->leader)) {
				$this->leaderList = new UserList();
				$this->leaderList->getConditionBuilder()->add('username IN (?)', [$this->leader]);
				$this->leaderList->readObjects();
				
				$tmp = [];
				foreach ($this->leaderList as $user) {
					$tmp[mb_strtolower($user->username)] = $user->username;
				}
				
				$difference = array_diff_key($this->leader, $tmp);
				if (!empty($difference)) {
					WCF::getTPL()->assign([
						'unknownLeader' => $difference
					]);
					throw new UserInputException('leader', 'notFound');
				}
				
				$this->leader = $tmp;
			}
			
			// moderated, closed moderated
			if (($this->groupType == 6 || $this->groupType == 7) && empty($this->leader)) {
				throw new UserInputException('leader');
			}
		}
		catch (UserInputException $e) {
			$this->eventObj->errorType[$e->getField()] = $e->getType();
		}
	}
}
