<?php
namespace wcf\page;
use wcf\data\user\group\application\UserGroupApplicationList;
use wcf\data\user\group\ModeratedUserGroup;
use wcf\data\user\group\UserGroup;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\menu\user\UserMenu;
use wcf\system\page\PageLocationManager;
use wcf\system\WCF;

/**
 * Lists applications for the group.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\Page
 * 
 * @property	UserGroupApplicationList	$objectList
 */
class UserGroupManageApplicationListPage extends SortablePage {
	/**
	 * @inheritDoc
	 */
	public $defaultSortField = 'time';
	
	/**
	 * @inheritDoc
	 */
	public $defaultSortOrder = 'DESC';
	
	/**
	 * @inheritDoc
	 */
	public $validSortFields = ['applicationID', 'applicant', 'username', 'time', 'status'];
	
	/**
	 * @inheritDoc
	 */
	public $objectListClassName = UserGroupApplicationList::class;
	
	/**
	 * @inheritDoc
	 */
	public $loginRequired = true;
	
	/**
	 * given groupID
	 * @var	integer
	 */
	public $groupID = 0;
	
	/**
	 * instance of the group with the given id
	 * @var	ModeratedUserGroup
	 */
	public $group;
	
	/**
	 * @inheritDoc
	 */
	protected function initObjectList() {
		parent::initObjectList();
		
		$this->objectList->getConditionBuilder()->add('groupID = ?', [$this->group->groupID]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		parent::readParameters();
		
		if (isset($_GET['id'])) $this->groupID = intval($_GET['id']);
		$this->group = new ModeratedUserGroup(new UserGroup($this->groupID));
		if (!$this->group->groupID) throw new IllegalLinkException();
		if (!$this->group->isLeader()) throw new PermissionDeniedException();
		
		// moderated, closed moderated
		if ($this->group->groupType != 6 && $this->group->groupType != 7) {
			throw new IllegalLinkException();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign([
			'group' => $this->group->getDecoratedObject()
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function show() {
		// set active tab
		UserMenu::getInstance()->setActiveMenuItem('wcf.user.menu.moderatedUserGroups');
		
		// add breadcrumbs
		PageLocationManager::getInstance()->addParentLocation('com.woltlab.wcf.moderatedUserGroup.ModeratedUserGroupList');
		
		parent::show();
	}
}
