<?php
namespace wcf\page;
use wcf\data\user\group\application\UserGroupApplicationList;
use wcf\data\user\group\UserGroup;
use wcf\data\user\group\UserGroupList;
use wcf\data\user\UserList;
use wcf\system\exception\IllegalLinkException;
use wcf\system\menu\user\UserMenu;
use wcf\system\WCF;

/**
 * Lists open, moderated and closed moderated groups.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\Page
 */
class UserGroupListPage extends \wcf\acp\page\UserGroupListPage {
	/**
	 * @inheritDoc
	 */
	public $activeMenuItem = '';
	
	/**
	 * @inheritDoc
	 */
	public $loginRequired = true;
	
	/**
	 * @inheritDoc
	 */
	public $neededPermissions = [];
	
	/**
	 * list of applications for the current user
	 * @var	UserGroupApplicationList
	 */
	public $applicationList;
	
	/**
	 * @inheritDoc
	 */
	public $itemsPerPage = 7;
	
	/**
	 * list of group leaders
	 * @var	UserList
	 */
	public $leaderList;
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	protected function initObjectList() {
		SortablePage::initObjectList();
		
		$userGroupList = new UserGroupList();
		$userGroupList->getConditionBuilder()->add('user_group.groupType NOT IN (?)', [[
			UserGroup::EVERYONE,
			UserGroup::GUESTS,
			UserGroup::USERS,
			UserGroup::OTHER
		]]);
		$userGroupList->readObjects();
		$validGroups = [0];
		// check whether there are any admin groups
		foreach ($userGroupList as $userGroup) {
			if (!$userGroup->isAdminGroup()) $validGroups[] = $userGroup->groupID;
		}
		
		// sort out groups that cannot be joined
		$this->objectList->getConditionBuilder()->add('user_group.groupID IN (?)', [$validGroups]);
		
		// read applications
		$this->applicationList = new UserGroupApplicationList();
		$this->applicationList->getConditionBuilder()->add('applicant = ?', [WCF::getUser()->userID]);
		
		// read group leaders
		$this->leaderList = new UserList();
		$this->leaderList->sqlJoins .= " INNER JOIN wcf".WCF_N."_user_group_leader group_leader ON user_table.userID = group_leader.leaderID";
	}
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		if (!count($this->objectList)) throw new IllegalLinkException();
		
		// assign applications to groups
		$this->applicationList->readObjects();
		foreach ($this->applicationList as $application) {
			if (($group = $this->objectList->search($application->groupID)) === null) continue;
			
			/** @noinspection PhpUndefinedFieldInspection */
			$group->application = $application;
		}
		
		// assign leaders to groups
		$this->leaderList->readObjects();
		$sql = "SELECT	*
			FROM	wcf".WCF_N."_user_group_leader";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute();
		while ($row = $statement->fetchArray()) {
			if (($group = $this->objectList->search($row['groupID'])) === null) continue;
			
			/** @noinspection PhpUndefinedFieldInspection */
			if (!is_array($group->leader)) {
				/** @noinspection PhpUndefinedFieldInspection */
				$group->leader = [];
			}
			/** @noinspection PhpUndefinedFieldInspection */
			$group->leader[] = $this->leaderList->search($row['leaderID']);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function show() {
		// set active tab
		UserMenu::getInstance()->setActiveMenuItem('wcf.user.menu.accessibleUserGroups');
		
		parent::show();
	}
}
