<?php
namespace wcf\form;
use wcf\data\user\group\application\UserGroupApplication;
use wcf\data\user\group\application\UserGroupApplicationAction;
use wcf\data\user\group\ModeratedUserGroup;
use wcf\data\user\group\ModeratedUserGroupAction;
use wcf\data\user\User;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\menu\user\UserMenu;
use wcf\system\page\PageLocationManager;
use wcf\system\request\LinkHandler;
use wcf\system\user\notification\object\UserGroupApplicationUserNotificationObject;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\WCF;
use wcf\util\HeaderUtil;
use wcf\util\StringUtil;

/**
 * Allows the group leader to edit the status of an application.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\Form
 */
class UserGroupManageApplicationEditForm extends UserGroupApplicationEditForm {
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		try {
			parent::readParameters();
		}
		catch (PermissionDeniedException $e) { }
		
		// check if current user is group leader
		$moderatedUserGroup = new ModeratedUserGroup($this->group);
		if (!$moderatedUserGroup->isLeader()) throw new PermissionDeniedException();
	}
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		if (empty($_POST)) {
			$this->status = $this->application->status == UserGroupApplication::STATUS_NEW ? UserGroupApplication::STATUS_IN_PROCESS : $this->application->status;
			$this->reply = $this->application->reply;
		}
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function show() {
		// set active tab
		UserMenu::getInstance()->setActiveMenuItem('wcf.user.menu.moderatedUserGroups');
		
		// add breadcrumbs
		PageLocationManager::getInstance()->addParentLocation('com.woltlab.wcf.moderatedUserGroup.UserGroupManageApplicationList', $this->group->groupID, new ModeratedUserGroup($this->group));
		PageLocationManager::getInstance()->addParentLocation('com.woltlab.wcf.moderatedUserGroup.ModeratedUserGroupList');
		
		AbstractForm::show();
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign([
			'isGroupLeader' => true
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function readFormParameters() {
		parent::readFormParameters();
		
		if (isset($_POST['status'])) $this->status = intval($_POST['status']);
		if (isset($_POST['reply'])) $this->reply = StringUtil::trim($_POST['reply']);
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function validate() {
		AbstractForm::validate();
		
		switch ($this->status) {
			case UserGroupApplication::STATUS_IN_PROCESS:
			case UserGroupApplication::STATUS_ACCEPTED:
			case UserGroupApplication::STATUS_REJECTED:
			case UserGroupApplication::STATUS_CLOSED:
				// they are all fine
			break;
			default:
				throw new UserInputException('status');
			break;
		}
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function save() {
		AbstractForm::save();
		
		if ($this->application->status != UserGroupApplication::STATUS_ACCEPTED) {
			$this->objectAction = new UserGroupApplicationAction([$this->application->applicationID], 'update', [
				'data' => array_merge($this->additionalFields, [
					'time' => TIME_NOW,
					'status' => $this->status,
					'reply' => $this->reply
				])
			]);
			$this->objectAction->executeAction();
			
			if ($this->status != $this->application->status) {
				// reload application object with updated data
				$application = new UserGroupApplication($this->application->applicationID);
				
				switch ($this->status) {
					case UserGroupApplication::STATUS_ACCEPTED:
						$applicant = new User($application->applicant);
						$groupAction = new ModeratedUserGroupAction([$this->group], 'addMember', ['user' => $applicant]);
						$groupAction->executeAction();
						
						UserNotificationHandler::getInstance()->fireEvent(
							'edited',
							'com.woltlab.wcf.moderatedUserGroup.application',
							new UserGroupApplicationUserNotificationObject($application),
							[$application->applicant],
							[
								'groupID' => $this->group->groupID,
								'newStatus' => 'accepted',
								'authorID' => WCF::getUser()->userID,
								'authorName' => WCF::getUser()->username
							]
						);
					break;
					
					case UserGroupApplication::STATUS_REJECTED:
						UserNotificationHandler::getInstance()->fireEvent(
							'edited',
							'com.woltlab.wcf.moderatedUserGroup.application',
							new UserGroupApplicationUserNotificationObject($application),
							[$application->applicant],
							[
								'groupID' => $this->group->groupID,
								'newStatus' => 'rejected',
								'authorID' => WCF::getUser()->userID,
								'authorName' => WCF::getUser()->username
							]
						);
					break;
					
					case UserGroupApplication::STATUS_CLOSED:
						UserNotificationHandler::getInstance()->fireEvent(
							'edited',
							'com.woltlab.wcf.moderatedUserGroup.application',
							new UserGroupApplicationUserNotificationObject($application),
							[$application->applicant],
							[
								'groupID' => $this->group->groupID,
								'newStatus' => 'closed',
								'authorID' => WCF::getUser()->userID,
								'authorName' => WCF::getUser()->username
							]
						);
					break;
				}
			}
			
			$this->saved();
		}
		else {
			throw new PermissionDeniedException();
		}
		
		// forward to user application list page
		HeaderUtil::redirect(LinkHandler::getInstance()->getLink('UserGroupManageApplicationList', ['id' => $this->group->groupID]));
		exit;
	}
}
