<?php
namespace wcf\form;
use wcf\data\user\group\application\UserGroupApplication;
use wcf\data\user\group\application\UserGroupApplicationAction;
use wcf\data\user\group\UserGroup;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\menu\user\UserMenu;
use wcf\system\page\PageLocationManager;
use wcf\system\request\LinkHandler;
use wcf\system\user\notification\object\UserGroupApplicationUserNotificationObject;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\WCF;
use wcf\util\HeaderUtil;
use wcf\util\StringUtil;

/**
 * Allows the user to apply for a moderated user group.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\Form
 */
class UserGroupApplyForm extends AbstractForm {
	/**
	 * @inheritDoc
	 */
	public $loginRequired = true;
	
	/**
	 * given groupID
	 * @var	integer
	 */
	public $groupID = 0;
	
	/**
	 * instance of the group with the given id
	 * @var	UserGroup
	 */
	public $group;
	
	/**
	 * given reason for the application
	 * @var	string
	 */
	public $reason = '';
	
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		parent::readParameters();
		
		if (isset($_GET['id'])) $this->groupID = intval($_GET['id']);
		$this->group = new UserGroup($this->groupID);
		if (!$this->group->groupID) throw new IllegalLinkException();
		
		// moderated, closed moderated
		if ($this->group->groupType != 6 && $this->group->groupType != 7) {
			throw new PermissionDeniedException();
		}
		
		if ($this->group->isMember()) {
			throw new PermissionDeniedException();
		}
		
		if ($this->group->isAdminGroup()) {
			throw new PermissionDeniedException();
		}
		
		// check whether there is an existing application
		if (UserGroupApplication::getApplicationByApplicantAndGroup(WCF::getUser()->userID, $this->group->groupID)->applicationID) {
			throw new PermissionDeniedException();
		}
		
		// check account status
		if (WCF::getUser()->activationCode) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readFormParameters() {
		parent::readFormParameters();
		
		if (isset($_POST['reason'])) $this->reason = StringUtil::trim($_POST['reason']);
	}
	
	/**
	 * @inheritDoc
	 */
	public function validate() {
		parent::validate();
		
		// reason
		if (empty($this->reason)) {
			throw new UserInputException('reason');
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign([
			'group' => $this->group,
			'reason' => $this->reason
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function show() {
		// set active tab
		UserMenu::getInstance()->setActiveMenuItem('wcf.user.menu.accessibleUserGroups');
		
		// add breadcrumbs
		PageLocationManager::getInstance()->addParentLocation('com.woltlab.wcf.moderatedUserGroup.UserGroupList');
		
		parent::show();
	}
	
	/**
	 * @inheritDoc
	 */
	public function save() {
		parent::save();
		
		$this->objectAction = new UserGroupApplicationAction([], 'create', [
			'data' => array_merge($this->additionalFields, [
				'applicant' => WCF::getUser()->userID,
				'username' => WCF::getUser()->username,
				'groupID' => $this->group->groupID,
				'time' => TIME_NOW,
				'reason' => $this->reason
			])
		]);
		$this->objectAction->executeAction();
		$returnValues = $this->objectAction->getReturnValues();
		$application = $returnValues['returnValues'];
		
		$sql = "SELECT	leaderID
			FROM	wcf".WCF_N."_user_group_leader
			WHERE	groupID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$this->group->groupID]);
		$groupLeaders = $statement->fetchAll(\PDO::FETCH_COLUMN);
		
		if (!empty($groupLeaders)) {
			UserNotificationHandler::getInstance()->fireEvent(
				'created',
				'com.woltlab.wcf.moderatedUserGroup.application',
				new UserGroupApplicationUserNotificationObject($application),
				$groupLeaders,
				['groupID' => $this->group->groupID]
			);
		}
		
		$this->saved();
		
		// forward to user group list page
		HeaderUtil::redirect(LinkHandler::getInstance()->getLink('UserGroupList'));
		exit;
	}
}
