<?php
namespace wcf\form;
use wcf\data\comment\StructuredCommentList;
use wcf\data\modification\log\ViewableUserGroupApplicationModificationLogList;
use wcf\data\user\group\application\UserGroupApplication;
use wcf\data\user\group\application\UserGroupApplicationAction;
use wcf\data\user\group\UserGroup;
use wcf\system\comment\manager\ICommentManager;
use wcf\system\comment\CommentHandler;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\menu\user\UserMenu;
use wcf\system\page\PageLocationManager;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;
use wcf\util\HeaderUtil;

/**
 * Allows the user to edit his application.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\Form
 */
class UserGroupApplicationEditForm extends UserGroupApplyForm {
	/**
	 * @inheritDoc
	 */
	public $templateName = 'userGroupApply';
	
	/**
	 * given applicationID
	 * @var	integer
	 */
	public $applicationID = 0;
	
	/**
	 * instance of the application with the given id
	 * @var	UserGroupApplication
	 */
	public $application;
	
	/**
	 * application status
	 * @var	integer
	 */
	public $status = 0;
	
	/**
	 * reply
	 * @var	string
	 */
	public $reply = '';
	
	/**
	 * modification log
	 * @var	ViewableUserGroupApplicationModificationLogList
	 */
	public $modificationLog;
	
	/**
	 * comment object type id
	 * @var	integer
	 */
	public $commentObjectTypeID = 0;
	
	/**
	 * comment manager object
	 * @var	ICommentManager
	 */
	public $commentManager;
	
	/**
	 * list of comments
	 * @var	StructuredCommentList
	 */
	public $commentList;
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		AbstractForm::readParameters();
		
		if (isset($_GET['id'])) $this->applicationID = intval($_GET['id']);
		$this->application = new UserGroupApplication($this->applicationID);
		if (!$this->application->applicationID) throw new IllegalLinkException();
		
		$this->group = new UserGroup($this->application->groupID);
		
		if ($this->application->applicant != WCF::getUser()->userID) {
			throw new PermissionDeniedException();
		}
		
		if ($this->application->status == UserGroupApplication::STATUS_ACCEPTED || $this->application->status == UserGroupApplication::STATUS_CLOSED) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		if (empty($_POST)) {
			$this->reason = $this->application->reason;
		}
		$this->status = $this->application->status;
		$this->reply = $this->application->reply;
		
		$this->modificationLog = new ViewableUserGroupApplicationModificationLogList();
		$this->modificationLog->setApplicationData([$this->application->applicationID]);
		$this->modificationLog->readObjects();
		
		$this->commentObjectTypeID = CommentHandler::getInstance()->getObjectTypeID('com.woltlab.wcf.moderatedUserGroup.application');
		$this->commentManager = CommentHandler::getInstance()->getObjectType($this->commentObjectTypeID)->getProcessor();
		$this->commentList = CommentHandler::getInstance()->getCommentList($this->commentManager, $this->commentObjectTypeID, $this->applicationID);
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign([
			'application' => $this->application,
			'status' => $this->status,
			'reply' => $this->reply,
			'modificationLog' => $this->modificationLog,
			'commentCanAdd' => true,
			'commentList' => $this->commentList,
			'commentObjectTypeID' => $this->commentObjectTypeID,
			'lastCommentTime' => $this->commentList ? $this->commentList->getMinCommentTime() : 0
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function save() {
		AbstractForm::save();
		
		$this->objectAction = new UserGroupApplicationAction([$this->application->applicationID], 'update', [
			'data' => array_merge($this->additionalFields, [
				'time' => TIME_NOW,
				'reason' => $this->reason
			])
		]);
		$this->objectAction->executeAction();
		$this->saved();
		
		// forward to user group list page
		HeaderUtil::redirect(LinkHandler::getInstance()->getLink('UserGroupList'));
		exit;
	}
	
	/**
	 * @inheritDoc
	 */
	public function show() {
		// set active tab
		UserMenu::getInstance()->setActiveMenuItem('wcf.user.menu.moderatedUserGroups');
		
		// add breadcrumbs
		PageLocationManager::getInstance()->addParentLocation('com.woltlab.wcf.moderatedUserGroup.ModeratedUserGroupList');
		
		parent::show();
	}
}
