<?php
namespace wcf\data\user\group\application;
use wcf\data\user\group\ModeratedUserGroup;
use wcf\data\user\group\UserGroup;
use wcf\data\user\UserProfile;
use wcf\data\DatabaseObject;
use wcf\data\ILinkableObject;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;

/**
 * Represents an application to a moderated user group.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\Data\User\Group\Application
 * 
 * @property-read	integer		$applicationID		unique id of the application to a moderated user group
 * @property-read	integer		$applicant		id of the user who has created the application
 * @property-read	string		$username		name of the user who has created the application
 * @property-read	integer		$groupID		id of the user group for which the user applies
 * @property-read	integer		$time			timestamp at which the application has been created
 * @property-read	string		$reason			reason for the application (written by the applicant)
 * @property-read	string		$reply			reply to the application (written by a user group leader)
 * @property-read	integer		$status			status of the application (see `UserGroupApplication::STATUS_*` constants)
 */
class UserGroupApplication extends DatabaseObject implements ILinkableObject {
	/**
	 * moderated user group the application is for
	 * @var	ModeratedUserGroup|null
	 */
	protected $group;
	
	/**
	 * user profile of the applicant
	 * @var	UserProfile|null
	 */
	protected $userProfile;
	
	/**
	 * default status
	 * @var	integer
	 */
	const STATUS_NEW = 0;
	
	/**
	 * the application is being processed by a group leader
	 * @var	integer
	 */
	const STATUS_IN_PROCESS = 1;
	
	/**
	 * the application was accepted
	 * @var	integer
	 */
	const STATUS_ACCEPTED = 2;
	
	/**
	 * the application was rejected, but the applicant may edit it
	 * @var	integer
	 */
	const STATUS_REJECTED = 3;
	
	/**
	 * the applicant may no longer edit the application
	 * @var	integer
	 */
	const STATUS_CLOSED = 5;
	
	/**
	 * @inheritDoc
	 */
	protected static $databaseTableName = 'user_group_application';
	
	/**
	 * @inheritDoc
	 */
	protected static $databaseTableIndexName = 'applicationID';
	
	/**
	 * Returns the moderated user group the application is for.
	 * 
	 * @return	ModeratedUserGroup
	 */
	public function getGroup() {
		if ($this->group === null) {
			$this->group = new ModeratedUserGroup(UserGroup::getGroupByID($this->groupID));
		}
		
		return $this->group;
	}
	
	/**
	 * Returns the user profile of the applicant.
	 * 
	 * @return	UserProfile
	 */
	public function getUserProfile() {
		if ($this->userProfile === null) {
			$this->userProfile = UserProfileRuntimeCache::getInstance()->getObject($this->applicant);
		}
		
		return $this->userProfile;
	}
	
	/**
	 * Returns the application matching applicant and groupID.
	 * 
	 * @param	integer		$applicant
	 * @param	integer		$groupID
	 * @return	UserGroupApplication
	 */
	public static function getApplicationByApplicantAndGroup($applicant, $groupID) {
		$sql = "SELECT	*
			FROM	wcf".WCF_N."_user_group_application
			WHERE		applicant = ?
				AND	groupID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$applicant, $groupID]);
		$row = $statement->fetchArray();
		if(!$row) $row = [];
		
		return new self(null, $row);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getLink() {
		return LinkHandler::getInstance()->getLink('UserGroupManageApplicationEdit', [
			'id' => $this->applicationID,
			'forceFrontend' => true
		]);
	}
}
