<?php
namespace wcf\data\user\group;
use wcf\data\user\User;
use wcf\data\user\UserProfileAction;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;

/**
 * Executes moderated user group-related actions.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2018 WoltLab GmbH
 * @license	GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @package	WoltLabSuite\Core\Data\User\Group
 */
class ModeratedUserGroupAction extends UserGroupAction {
	/**
	 * user object
	 * @var	User
	 */
	protected $user;
	
	/**
	 * Adds a user to this group.
	 */
	public function addMember() {
		if (empty($this->objects)) {
			$this->readObjects();
		}
		
		/** @var User $user */
		$user = $this->parameters['user'];
		$groups = $user->getGroupIDs();
		
		foreach ($this->getObjects() as $group) {
			$groups[] = $group->groupID;
		}
		
		$data = [
			'groups' => $groups
		];
		$userAction = new UserProfileAction([$user->userID], 'update', $data);
		$userAction->executeAction();
		
		$updateUserRankAction = new UserProfileAction([$user->userID], 'updateUserRank');
		$updateUserRankAction->executeAction();
		
		$updateUserOnlineMarkingAction = new UserProfileAction([$user->userID], 'updateUserOnlineMarking');
		$updateUserOnlineMarkingAction->executeAction();
	}
	
	/**
	 * Adds a user to this group.
	 */
	public function addMemberByName() {
		if (empty($this->objects)) {
			$this->readObjects();
		}
		
		if ($this->user === null) {
			$this->user = User::getUserByUsername($this->parameters['username']);
		}
		
		$action = new ModeratedUserGroupAction($this->objects, 'addMember', ['user' => $this->user]);
		$action->executeAction();
	}
	
	/**
	 * Validates the 'addMemberByName' action.
	 */
	public function validateAddMemberByName() {
		$this->readString('username');
		
		$this->readObjects();
		
		if (empty($this->objects)) {
			throw new UserInputException('objectIDs');
		}
		
		foreach ($this->getObjects() as $group) {
			$moderatedGroup = new ModeratedUserGroup($group->getDecoratedObject());
			if (!$moderatedGroup->isLeader()) {
				throw new PermissionDeniedException();
			}
		}
		
		// validate username
		$this->user = User::getUserByUsername($this->parameters['username']);
		if (!$this->user->userID) {
			throw new UserInputException('username', 'notFound');
		}
		
		foreach ($this->getObjects() as $group) {
			if (in_array($group->groupID, $this->user->getGroupIDs())) {
				throw new UserInputException('username', 'alreadyMember');
			}
		}
	}
}
