<?php
namespace wcf\action;
use wcf\data\user\group\application\UserGroupApplication;
use wcf\data\user\group\application\UserGroupApplicationEditor;
use wcf\data\user\group\ModeratedUserGroup;
use wcf\data\user\group\UserGroup;
use wcf\data\user\User;
use wcf\data\user\UserProfileAction;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\request\LinkHandler;
use wcf\util\HeaderUtil;

/**
 * Allows the group leader to kick members.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Core\Action
 */
class UserGroupManageKickMemberAction extends AbstractSecureAction {
	/**
	 * @inheritDoc
	 */
	public $loginRequired = true;
	
	/**
	 * database object action
	 * @var	\wcf\data\AbstractDatabaseObjectAction
	 */
	public $objectAction;
	
	/**
	 * given groupID
	 * @var	integer
	 */
	public $groupID = 0;
	
	/**
	 * instance of the group with the given id
	 * @var	UserGroup
	 */
	public $group;
	
	/**
	 * given memberID
	 * @var	integer
	 */
	public $memberID = 0;
	
	/**
	 * instance of the user with the given id
	 * @var	User
	 */
	public $member;
	
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		parent::readParameters();
		
		if (isset($_GET['id'])) $this->groupID = intval($_GET['id']);
		$this->group = new UserGroup($this->groupID);
		if (!$this->group->groupID) throw new IllegalLinkException();
		
		if (isset($_GET['memberID'])) $this->memberID = intval($_GET['memberID']);
		$this->member = new User($this->memberID);
		if (!$this->member->userID) throw new IllegalLinkException();
	}
	
	/**
	 * @inheritDoc
	 */
	public function execute() {
		parent::execute();
		
		$moderatedUserGroup = new ModeratedUserGroup($this->group);
		if (!$moderatedUserGroup->isLeader()) {
			throw new PermissionDeniedException();
		}
		
		if (!$this->group->isMember($this->member)) {
			throw new IllegalLinkException();
		}
		
		$data = [
			'removeGroups' => [$this->group->groupID]
		];
		$this->objectAction = new UserProfileAction([$this->member->userID], 'update', $data);
		$this->objectAction->executeAction();
		
		$updateUserRankAction = new UserProfileAction([$this->member->userID], 'updateUserRank');
		$updateUserRankAction->executeAction();
		
		$updateUserOnlineMarkingAction = new UserProfileAction([$this->member->userID], 'updateUserOnlineMarking');
		$updateUserOnlineMarkingAction->executeAction();
		
		$applicationEditor = new UserGroupApplicationEditor(UserGroupApplication::getApplicationByApplicantAndGroup($this->member->userID, $this->group->groupID));
		$applicationEditor->delete();
		
		$this->executed();
		
		// forward to member list page
		HeaderUtil::redirect(LinkHandler::getInstance()->getLink('UserGroupManageMemberList', ['id' => $this->group->groupID]));
		exit;
	}
}
