<?php
namespace wbb\system\worker;
use wbb\data\thread\ThreadList;
use wcf\data\object\type\ObjectTypeCache;
use wbb\data\thread\ThreadEditor;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\user\activity\point\UserActivityPointHandler;
use wcf\system\worker\AbstractRebuildDataWorker;
use wcf\system\WCF;

/**
 * Worker implementation for updating threads.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Worker
 * 
 * @method	ThreadList	getObjectList()
 */
class ThreadRebuildDataWorker extends AbstractRebuildDataWorker {
	/**
	 * @inheritDoc
	 */
	protected $objectListClassName = ThreadList::class;
	
	/**
	 * @inheritDoc
	 */
	protected $limit = 500;
	
	/**
	 * @inheritDoc
	 */
	protected function initObjectList() {
		parent::initObjectList();
		
		$this->objectList->sqlOrderBy = 'thread.threadID';
	}
	
	/**
	 * @inheritDoc
	 */
	public function execute() {
		parent::execute();
		
		if (!$this->loopCount) {
			// reset activity points
			UserActivityPointHandler::getInstance()->reset('com.woltlab.wbb.activityPointEvent.thread');
		}
		
		if (!count($this->objectList)) {
			return;
		}
		
		// get label status
		$hasLabels = [];
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.label.object', 'com.woltlab.wbb.thread')->objectTypeID]);
		$conditionBuilder->add('objectID IN (?)', [$this->objectList->getObjectIDs()]);
		$sql = "SELECT	DISTINCT objectID
			FROM	wcf".WCF_N."_label_object
			".$conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		while ($row = $statement->fetchArray()) {
			$hasLabels[$row['objectID']] = 1;
		}
		
		// prepare statements
		$sql = "SELECT		postID, time, userID, username, cumulativeLikes
			FROM		wbb".WCF_N."_post
			WHERE		threadID = ?
					AND isDeleted = ?
					AND isDisabled = ?
			ORDER BY	time ASC, postID ASC";
		$firstPostStatement = WCF::getDB()->prepareStatement($sql, 1);
		$sql = "SELECT		postID, time, userID, username
			FROM		wbb".WCF_N."_post
			WHERE		threadID = ?
					AND isDeleted = ?
					AND isDisabled = ?
			ORDER BY	time DESC, postID DESC";
		$lastPostStatement = WCF::getDB()->prepareStatement($sql, 1);
		$sql = "SELECT	COUNT(*) AS posts,
				SUM(attachments) AS attachments,
				SUM(CASE WHEN pollID <> 0 THEN 1 ELSE 0 END) AS polls
			FROM	wbb".WCF_N."_post
			WHERE	threadID = ?
				AND isDeleted = ?
				AND isDisabled = ?";
		$statsStatement = WCF::getDB()->prepareStatement($sql);
		
		$userToItems = [];
		WCF::getDB()->beginTransaction();
		foreach ($this->objectList as $thread) {
			if ($thread->movedThreadID) continue;
			$editor = new ThreadEditor($thread);
			$data = [];
			
			// get first post
			$firstPostStatement->execute([$thread->threadID, $thread->isDeleted, $thread->isDisabled]);
			if (($row = $firstPostStatement->fetchSingleRow()) !== false) {
				$data['firstPostID'] = $row['postID'];
				$data['time'] = $row['time'];
				$data['userID'] = $row['userID'];
				$data['username'] = $row['username'];
				$data['cumulativeLikes'] = $row['cumulativeLikes'];
			}
			
			// get last post
			$lastPostStatement->execute([$thread->threadID, $thread->isDeleted, $thread->isDisabled]);
			if (($row = $lastPostStatement->fetchSingleRow()) !== false) {
				$data['lastPostID'] = $row['postID'];
				$data['lastPostTime'] = $row['time'];
				$data['lastPosterID'] = $row['userID'];
				$data['lastPoster'] = $row['username'];
			}
			
			// get stats
			$statsStatement->execute([$thread->threadID, 0, 0]);
			$row = $statsStatement->fetchSingleRow();
			$data['replies'] = $row['posts'] ? $row['posts'] - 1 : 0;
			$data['attachments'] = $row['attachments'] ?: 0;
			$data['polls'] = $row['polls'] ?: 0;
			
			// label status
			if (isset($hasLabels[$thread->threadID])) {
				$data['hasLabels'] = 1;
			}
			
			$editor->update($data);
			
			// update activity points
			if (!$thread->isDisabled && $thread->userID && $thread->getBoard()->countUserPosts) {
				if (!isset($userToItems[$thread->userID])) {
					$userToItems[$thread->userID] = 0;
				}
				$userToItems[$thread->userID]++;
			}
		}
		WCF::getDB()->commitTransaction();
		
		// update activity points
		UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.thread', $userToItems, false);
	}
}
