<?php
namespace wbb\system\worker;
use wbb\data\post\PostAction;
use wbb\data\post\PostEditor;
use wbb\data\post\PostList;
use wbb\data\thread\ThreadList;
use wbb\system\cache\builder\StatsCacheBuilder;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\option\OptionEditor;
use wcf\system\bbcode\BBCodeHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\html\input\HtmlInputProcessor;
use wcf\system\message\embedded\object\MessageEmbeddedObjectManager;
use wcf\system\user\activity\point\UserActivityPointHandler;
use wcf\system\worker\AbstractRebuildDataWorker;
use wcf\system\WCF;

/**
 * Worker implementation for updating posts.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Worker
 * 
 * @method	PostList	getObjectList()
 */
class PostRebuildDataWorker extends AbstractRebuildDataWorker {
	/**
	 * @inheritDoc
	 */
	protected $limit = 500;
	
	/**
	 * @var HtmlInputProcessor
	 */
	protected $htmlInputProcessor;
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function countObjects() {
		if ($this->count === null) {
			$this->count = 0;
			$sql = "SELECT	MAX(postID) AS postID
				FROM	wbb".WCF_N."_post";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute();
			$row = $statement->fetchArray();
			if ($row !== false) $this->count = $row['postID'];
		}
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	protected function initObjectList() {
		$this->objectList = new PostList();
		$this->objectList->sqlOrderBy = 'post.postID';
	}
	
	/**
	 * @inheritDoc
	 */
	public function execute() {
		$this->objectList->getConditionBuilder()->add('post.postID BETWEEN ? AND ?', [$this->limit * $this->loopCount + 1, $this->limit * $this->loopCount + $this->limit]);
		
		parent::execute();
		
		if (!$this->loopCount) {
			// reset activity points
			UserActivityPointHandler::getInstance()->reset('com.woltlab.wbb.activityPointEvent.post');
			
			// update install date
			$sql = "SELECT	MIN(time)
				FROM	wbb".WCF_N."_post
				WHERE   time > ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([0]);
			$time = $statement->fetchSingleColumn();
				
			if (!WBB_INSTALL_DATE || ($time && $time < WBB_INSTALL_DATE)) {
				// set install date
				$sql = "UPDATE	wcf".WCF_N."_option
					SET	optionValue = ?
					WHERE	optionName = ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute([$time ?: TIME_NOW, 'wbb_install_date']);
				
				OptionEditor::resetCache();
				StatsCacheBuilder::getInstance()->reset();
			}
		}
		
		if (!count($this->objectList)) {
			return;
		}
		
		// fetch cumulative likes
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("objectTypeID = ?", [ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.like.likeableObject', 'com.woltlab.wbb.likeablePost')]);
		$conditions->add("objectID IN (?)", [$this->objectList->getObjectIDs()]);
		
		$sql = "SELECT	objectID, cumulativeLikes
			FROM	wcf".WCF_N."_like_object
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		$cumulativeLikes = $statement->fetchMap('objectID', 'cumulativeLikes');
		
		// fetch attachment counts
		$attachmentObjectType = ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.attachment.objectType', 'com.woltlab.wbb.post');
		$conditions = new PreparedStatementConditionBuilder($sql);
		$conditions->add("objectTypeID = ?", [$attachmentObjectType->objectTypeID]);
		$conditions->add("objectID IN (?)", [$this->objectList->getObjectIDs()]);
		
		$sql = "SELECT		objectID, COUNT(*) AS attachments
			FROM		wcf" . WCF_N . "_attachment
			".$conditions."
			GROUP BY        objectID";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		$attachments = $statement->fetchMap('objectID', 'attachments');
		
		// read associated threads
		$threadIDs = [];
		foreach ($this->objectList as $post) {
			$threadIDs[] = $post->threadID;
		}
		
		$threadList = new ThreadList();
		$threadList->setObjectIDs($threadIDs);
		$threadList->readObjects();
		$threads = $threadList->getObjects();
		
		// retrieve permissions
		$userIDs = [];
		foreach ($this->objectList as $object) {
			$userIDs[] = $object->userID;
		}
		$userPermissions = $this->getBulkUserPermissions($userIDs, ['user.message.disallowedBBCodes']);
		
		$itemsToUser = [];
		$disablePosts = [];
		WCF::getDB()->beginTransaction();
		foreach ($this->objectList as $post) {
			$post->setThread($threads[$post->threadID]);
			
			$editor = new PostEditor($post);
			$data = [];
			
			// disable post, if thread is disabled
			if ($post->getThread()->isDisabled && !$post->isDisabled) $disablePosts[] = $post;
			
			// count attachments
			$data['attachments'] = (isset($attachments[$post->postID])) ? $attachments[$post->postID] : 0;
			
			// update cumulative likes
			$data['cumulativeLikes'] = isset($cumulativeLikes[$post->postID]) ? $cumulativeLikes[$post->postID] : 0;
			
			BBCodeHandler::getInstance()->setDisallowedBBCodes(explode(',', $this->getBulkUserPermissionValue($userPermissions, $post->userID, 'user.message.disallowedBBCodes')));
			
			// update message
			if (!$post->enableHtml) {
				$this->getHtmlInputProcessor()->process($post->message, 'com.woltlab.wbb.post', $post->postID, true);
				$data['message'] = $this->getHtmlInputProcessor()->getHtml();
				$data['enableHtml'] = 1;
			}
			else {
				$this->getHtmlInputProcessor()->reprocess($post->message, 'com.woltlab.wbb.post', $post->postID);
				$data['message'] = $this->getHtmlInputProcessor()->getHtml();
			}
			
			if (MessageEmbeddedObjectManager::getInstance()->registerObjects($this->getHtmlInputProcessor(), true)) {
				$data['hasEmbeddedObjects'] = 1;
			}
			else {
				$data['hasEmbeddedObjects'] = 0;
			}
			
			// update data
			$editor->update($data);
			
			if (!$post->isDisabled && $post->userID && $post->getThread()->getBoard()->countUserPosts && !$post->isFirstPost()) {
				if (!isset($itemsToUser[$post->userID])) {
					$itemsToUser[$post->userID] = 0;
				}
				$itemsToUser[$post->userID]++;
			}
		}
		
		MessageEmbeddedObjectManager::getInstance()->commitBulkOperation();
		
		WCF::getDB()->commitTransaction();
		
		// update activity points
		UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.post', $itemsToUser, false);
		
		if (!empty($disablePosts)) {
			$postAction = new PostAction($disablePosts, 'disable', [
				'ignoreThreads' => true,
				'isBulkProcessing' => true
			]);
			$postAction->executeAction();
		}
	}
	
	/**
	 * @return HtmlInputProcessor
	 */
	protected function getHtmlInputProcessor() {
		if ($this->htmlInputProcessor === null) {
			$this->htmlInputProcessor = new HtmlInputProcessor();
		}
		
		return $this->htmlInputProcessor;
	}
}
