<?php
namespace wbb\system\user\notification\event;
use wbb\data\post\LikeablePost;
use wbb\system\cache\runtime\PostRuntimeCache;
use wbb\system\cache\runtime\ThreadRuntimeCache;
use wcf\data\user\UserProfile;
use wcf\system\user\notification\event\AbstractSharedUserNotificationEvent;
use wcf\system\user\notification\event\ITestableUserNotificationEvent;
use wcf\system\user\notification\event\TReactionUserNotificationEvent;
use wcf\system\user\notification\event\TTestableLikeUserNotificationEvent;
use wcf\system\user\notification\event\TTestableUserNotificationEvent;
use wcf\system\user\notification\object\IUserNotificationObject;
use wcf\system\user\notification\object\LikeUserNotificationObject;
use wcf\system\user\object\watch\UserObjectWatchHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\WCF;

/**
 * User notification event for post likes.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\User\Notification\Event
 * 
 * @method	LikeUserNotificationObject	getUserNotificationObject()
 */
class PostLikeUserNotificationEvent extends AbstractSharedUserNotificationEvent implements ITestableUserNotificationEvent {
	use TTestableLikeUserNotificationEvent {
		TTestableLikeUserNotificationEvent::canBeTriggeredByGuests insteadof TTestableUserNotificationEvent;
	}
	use TTestablePostRelatedUserNotificationEvent;
	use TTestableUserNotificationEvent;
	use TReactionUserNotificationEvent;
	
	/**
	 * @inheritDoc
	 */
	protected $stackable = true;
	
	/**
	 * @inheritDoc
	 */
	protected function prepare() {
		PostRuntimeCache::getInstance()->cacheObjectID($this->additionalData['objectID']);
		ThreadRuntimeCache::getInstance()->cacheObjectID($this->additionalData['threadID']);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTitle() {
		$count = count($this->getAuthors());
		if ($count > 1) {
			return $this->getLanguage()->getDynamicVariable('wbb.post.like.notification.title.stacked', [
				'count' => $count,
				'timesTriggered' => $this->notification->timesTriggered
			]);
		}
		
		return $this->getLanguage()->getDynamicVariable('wbb.post.like.notification.title');
	}
	
	/**
	 * @inheritDoc
	 */
	public function getMessage() {
		$post = PostRuntimeCache::getInstance()->getObject($this->additionalData['objectID']);
		if ($post->threadID == $this->additionalData['threadID']) {
			$thread = ThreadRuntimeCache::getInstance()->getObject($this->additionalData['threadID']);
		}
		else {
			$thread = ThreadRuntimeCache::getInstance()->getObject($post->threadID);
		}
		$post->setThread($thread);
		
		$authors = array_values($this->getAuthors());
		$count = count($authors);
		
		if ($count > 1) {
			return $this->getLanguage()->getDynamicVariable('wbb.post.like.notification.message.stacked', [
				'author' => $this->author,
				'authors' => $authors,
				'count' => $count,
				'others' => $count - 1,
				'post' => $post,
				'thread' => $thread,
				'reactions' => $this->getReactionsForAuthors()
			]);
		}
		
		return $this->getLanguage()->getDynamicVariable('wbb.post.like.notification.message', [
			'author' => $this->author,
			'thread' => $thread,
			'post' => $post,
			'userNotificationObject' => $this->getUserNotificationObject(),
			'reactions' => $this->getReactionsForAuthors()
		]);
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getEmailMessage($notificationType = 'instant') {
		// not supported
	}
	
	/**
	 * @inheritDoc
	 */
	public function getLink() {
		$post = PostRuntimeCache::getInstance()->getObject($this->additionalData['objectID']);
		if ($post->threadID == $this->additionalData['threadID']) {
			$thread = ThreadRuntimeCache::getInstance()->getObject($this->additionalData['threadID']);
		}
		else {
			$thread = ThreadRuntimeCache::getInstance()->getObject($post->threadID);
		}
		$post->setThread($thread);
		
		return $post->getLink();
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getEventHash() {
		return sha1($this->eventID . '-' . $this->additionalData['objectID']);
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function supportsEmailNotification() {
		return false;
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function checkAccess() {
		$post = PostRuntimeCache::getInstance()->getObject($this->additionalData['objectID']);
		if ($post->threadID == $this->additionalData['threadID']) {
			$thread = ThreadRuntimeCache::getInstance()->getObject($this->additionalData['threadID']);
		}
		else {
			$thread = ThreadRuntimeCache::getInstance()->getObject($post->threadID);
		}
		$post->setThread($thread);
		
		$returnValue = $post->canRead();
		if (!$post->getThread()->canRead()) {
			// remove subscription
			UserObjectWatchHandler::getInstance()->deleteObjects('com.woltlab.wbb.thread', [$post->threadID], [WCF::getUser()->userID]);
			
			// reset user storage
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedThreads');
			
			$returnValue = false;
		}
		
		return $returnValue;
	}
	
	/**
	 * @inheritDoc
	 */
	public static function getTestAdditionalData(IUserNotificationObject $object) {
		/** @var LikeablePost $likedObject */
		$likedObject = self::getTestLikeObject($object);
		
		return [
			'objectID' => $likedObject->postID,
			'threadID' => $likedObject->threadID
		];
	}
	
	/**
	 * @inheritDoc
	 * @since	3.1
	 */
	protected static function createTestLikeObject(UserProfile $recipient, UserProfile $author) {
		return new LikeablePost(self::createTestPost($recipient, $author));
	}
	
	/**
	 * @inheritDoc
	 * @since	3.1
	 */
	protected static function getTestLikeableObjectTypeName() {
		return 'com.woltlab.wbb.likeablePost';
	}
}
