<?php
namespace wbb\system\thread;
use wbb\data\thread\Thread;
use wbb\system\thread\editor\DefaultThreadEditor;
use wbb\system\thread\editor\LabelThreadEditor;
use wbb\system\thread\editor\TagThreadEditor;
use wcf\system\event\EventHandler;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\SingletonFactory;

/**
 * Handles thread-related actions.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Thread
 */
class ThreadHandler extends SingletonFactory {
	/**
	 * list of thread editors
	 * @var	mixed[][]
	 */
	public $editors = [
		'type' => [],
		'object' => []
	];
	
	/**
	 * @inheritDoc
	 */
	protected function init() {
		$this->editors = [
			// the order within the 'type' array represents the display order
			'type' => [
				'label',
				'default',
				'tag'
			],
			'object' => [
				'default' => new DefaultThreadEditor(),
				'label' => new LabelThreadEditor(),
				'tag' => new TagThreadEditor()
			]
		];
		
		EventHandler::getInstance()->fireAction($this, 'didInit');
	}
	
	/**
	 * Returns template for thread editor.
	 * 
	 * @param	integer		$threadID
	 * @return	string
	 */
	public function beginEdit($threadID) {
		$this->setThread($threadID);
		$template = '';
		
		foreach ($this->editors['type'] as $typeName) {
			$template .= $this->editors['object'][$typeName]->beginEdit();
		}
		
		return $template;
	}
	
	/**
	 * Saves editor data.
	 * 
	 * @param	integer		$threadID
	 * @param	mixed[]		$parameters
	 * @return	array
	 */
	public function saveEdit($threadID, array $parameters) {
		$this->setThread($threadID);
		
		// validate parameters prior to updating data
		$errors = [];
		foreach ($this->editors['type'] as $typeName) {
			if (isset($parameters[$typeName])) {
				try {
					$this->editors['object'][$typeName]->validate($parameters[$typeName]);
				}
				catch (UserInputException $e) {
					if (!isset($errors[$typeName])) {
						$errors[$typeName] = [];
					}
					
					$errors[$typeName][$e->getField()] = $e->getType();
				}
			}
		}
		
		// update data
		$forceReload = false;
		$returnValues = [];
		if (empty($errors)) {
			foreach ($this->editors['type'] as $typeName) {
				if (isset($parameters[$typeName])) {
					$returnValues[$typeName] = $this->editors['object'][$typeName]->saveEdit($parameters[$typeName]);
				}
			}
			
			foreach ($returnValues as $data) {
				if (isset($data['forceReload']) && $data['forceReload']) {
					$forceReload = true;
					break;
				}
			}
		}
		
		$returnValues['errors'] = $errors;
		$returnValues['forceReload'] = $forceReload;
		
		return $returnValues;
	}
	
	/**
	 * Loads and assigns a thread to loaded editors.
	 * 
	 * @param	integer		$threadID
	 * @throws	IllegalLinkException
	 * @throws	PermissionDeniedException
	 */
	protected function setThread($threadID) {
		$thread = new Thread($threadID);
		if (!$thread->threadID) {
			throw new IllegalLinkException();
		}
		else if (!$thread->canEdit()) {
			throw new PermissionDeniedException();
		}
		
		foreach ($this->editors['object'] as $editor) {
			$editor->setThread($thread);
		}
	}
}
