<?php
namespace wbb\system\search;
use wbb\data\board\BoardCache;
use wbb\data\board\SearchableBoardNodeList;
use wbb\data\post\SearchResultPostList;
use wbb\data\thread\SearchResultThreadList;
use wbb\data\thread\Thread;
use wbb\system\event\listener\SearchListener;
use wbb\system\WBBCore;
use wcf\form\IForm;
use wcf\form\SearchForm;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\language\LanguageFactory;
use wcf\system\search\AbstractSearchableObjectType;
use wcf\system\WCF;
use wcf\util\ArrayUtil;

/**
 * An implementation of ISearchableObjectType for searching in forum posts.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Search
 */
class PostSearch extends AbstractSearchableObjectType {
	/**
	 * message data cache
	 * @var	array
	 */
	public $messageCache = [];
	
	/**
	 * board ids
	 * @var	integer[]
	 */
	public $boardIDs = [];
	
	/**
	 * enables search for attachments
	 * @var	boolean
	 */
	public $findAttachments = 0;
	
	/**
	 * enables search for polls
	 * @var	boolean
	 */
	public $findPolls = 0;
	
	/**
	 * shows results as threads
	 * @var	boolean
	 */
	public $findThreads = WBB_SEARCH_FIND_THREADS;
	
	/**
	 * enables search for user threads
	 * @var	integer
	 */
	public $findUserThreads = 0;
	
	/**
	 * list of all boards
	 * @var	array
	 */
	public $boards = [];
	
	/**
	 * list of selected boards
	 * @var	array
	 */
	public $selectedBoards = [];
	
	/**
	 * id of the searched thread
	 * @var	integer
	 */
	public $threadID = 0;
	
	/**
	 * searched thread
	 * @var	Thread
	 */
	public $thread;
	
	/**
	 * @inheritDoc
	 */
	public function cacheObjects(array $objectIDs, array $additionalData = null) {
		if ($additionalData !== null && !empty($additionalData['findThreads'])) {
			SearchListener::$findThreads = true;
			$threadList = new SearchResultThreadList();
			$threadList->setObjectIDs($objectIDs);
			$threadList->readObjects();
			foreach ($threadList->getObjects() as $thread) {
				$this->messageCache[$thread->threadID] = $thread;
			}
		}
		else {
			$postList = new SearchResultPostList();
			$postList->setObjectIDs($objectIDs);
			$postList->readObjects();
			foreach ($postList->getObjects() as $post) {
				$this->messageCache[$post->postID] = $post;
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getObject($objectID) {
		if (isset($this->messageCache[$objectID])) return $this->messageCache[$objectID];
		return null;
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getFormTemplateName() {
		return 'searchPost';
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getJoins() {
		return "LEFT JOIN wbb".WCF_N."_thread thread ON (thread.threadID = ".$this->getTableName().".threadID)";
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTableName() {
		return 'wbb'.WCF_N.'_post';
	}
	
	/**
	 * @inheritDoc
	 */
	public function getIDFieldName() {
		return $this->getTableName().'.postID';
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getAdditionalData() {
		return [
			'findThreads' => $this->findThreads,
			'findAttachments' => $this->findAttachments,
			'findPolls' => $this->findPolls,
			'findUserThreads' => $this->findUserThreads,
			'boardIDs' => $this->boardIDs,
			'threadID' => $this->threadID
		];
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getConditions(IForm $form = null) {
		/** @var SearchForm $form */
		
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$this->readFormParameters();
		
		$boardIDs = $this->boardIDs;
		
		// get all boards from cache
		$this->boards = BoardCache::getInstance()->getBoards();
		$this->selectedBoards = [];
		
		// check whether the selected board does exist
		foreach ($boardIDs as $boardID) {
			if (!isset($this->boards[$boardID]) || !$this->boards[$boardID]->searchable) {
				throw new UserInputException('boardIDs', 'invalid');
			}
			
			if (!isset($this->selectedBoards[$boardID])) {
				$this->selectedBoards[$boardID] = $this->boards[$boardID];
				
				// include children
				$this->includeSubBoards($boardID);
			}
		}
		
		if (empty($this->selectedBoards)) {
			$this->selectedBoards = $this->boards;
		}
		
		// check permission of the active user
		foreach ($this->selectedBoards as $board) {
			if ($board->isIgnored() || !$board->getPermission() || !$board->getPermission('canEnterBoard') || !$board->getPermission('canReadThread') || !$board->searchable || ($board->isPrivate && !WCF::getUser()->userID)) {
				unset($this->selectedBoards[$board->boardID]);
			}
		}
		if (count($this->selectedBoards) == 0) {
			throw new PermissionDeniedException();
		}
		
		// get board ids
		$boardIDs = $privateBoardIDs = [];
		foreach ($this->selectedBoards as $board) {
			if ($board->isPrivate && !$board->canReadPrivateThreads()) {
				$privateBoardIDs[] = $board->boardID;
				continue;
			}
			
			$boardIDs[] = $board->boardID;
		}
		
		// board ids
		if (empty($privateBoardIDs)) {
			$conditionBuilder->add('thread.boardID IN (?)', [$boardIDs]);
		}
		else if (empty($boardIDs)) {
			$conditionBuilder->add('(thread.boardID IN (?) AND thread.userID = ?)', [$privateBoardIDs, WCF::getUser()->userID]);
		}
		else {
			$conditionBuilder->add('(thread.boardID IN (?) OR (thread.boardID IN (?) AND thread.userID = ?))', [$boardIDs, $privateBoardIDs, WCF::getUser()->userID]);
		}
		
		// find user threads
		if ($this->findUserThreads && $form !== null && ($userIDs = $form->getUserIDs())) {
			$conditionBuilder->add('thread.userID IN (?)', [$userIDs]);
		}
		
		if (!$this->findThreads) {
			// default conditions
			$conditionBuilder->add($this->getTableName().'.isDeleted = 0');
			$conditionBuilder->add($this->getTableName().'.isDisabled = 0');
			
			// find attachments
			if ($this->findAttachments) $conditionBuilder->add($this->getTableName().'.attachments > 0');
			// find polls
			if ($this->findPolls) $conditionBuilder->add($this->getTableName().'.pollID IS NOT NULL');
		}
		
		// language
		if (LanguageFactory::getInstance()->multilingualismEnabled() && count(WCF::getUser()->getLanguageIDs())) {
			$conditionBuilder->add('(thread.languageID IN (?) OR thread.languageID IS NULL)', [WCF::getUser()->getLanguageIDs()]);
		}
		
		// filter by thread
		if ($this->threadID) {
			$conditionBuilder->add($this->getTableName().'.threadID = ?', [$this->threadID]);
		}
		
		return $conditionBuilder;
	}
	
	/**
	 * Recursively adds sub boards to the list of selected boards.
	 * 
	 * @param	integer		$boardID
	 */
	private function includeSubBoards($boardID) {
		foreach (BoardCache::getInstance()->getChildIDs($boardID) as $childBoardID) {
			if (!isset($this->selectedBoards[$childBoardID])) {
				$this->selectedBoards[$childBoardID] = $this->boards[$childBoardID];
				
				// include children
				$this->includeSubBoards($childBoardID);
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function show(IForm $form = null) {
		/** @var SearchForm $form */
		
		// get searchable boards
		$boardNodeList = new SearchableBoardNodeList();
		$boardNodeList->readNodeTree();
		
		// get existing values
		if ($form !== null && isset($form->searchData['additionalData']['com.woltlab.wbb.post'])) {
			$this->boardIDs = $form->searchData['additionalData']['com.woltlab.wbb.post']['boardIDs'];
			$this->findAttachments = $form->searchData['additionalData']['com.woltlab.wbb.post']['findAttachments'];
			$this->findPolls = $form->searchData['additionalData']['com.woltlab.wbb.post']['findPolls'];
			$this->findThreads = $form->searchData['additionalData']['com.woltlab.wbb.post']['findThreads'];
			$this->findUserThreads = $form->searchData['additionalData']['com.woltlab.wbb.post']['findUserThreads'];
			
			// existence check for backwards compatibility
			if (isset($form->searchData['additionalData']['com.woltlab.wbb.post']['threadID'])) {
				$this->threadID = $form->searchData['additionalData']['com.woltlab.wbb.post']['threadID'];
				
				if ($this->threadID) {
					$this->thread = new Thread($this->threadID);
					if (!$this->thread->threadID || !$this->thread->canRead()) {
						$this->threadID = 0;
						$this->thread = null;
					}
				}
			}
		}
		
		// get default board ids
		if (isset($_GET['boardIDs']) && is_array($_GET['boardIDs'])) {
			$this->boardIDs = ArrayUtil::toIntegerArray($_GET['boardIDs']);
		}
		
		WCF::getTPL()->assign([
			'boardNodeList' => $boardNodeList->getNodeList(),
			'boardIDs' => $this->boardIDs,
			'findAttachments' => $this->findAttachments,
			'findPolls' => $this->findPolls,
			'findThreads' => $this->findThreads,
			'findUserThreads' => $this->findUserThreads,
			'searchedThread' => $this->thread
		]);
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getOuterSQLQuery($q, PreparedStatementConditionBuilder &$searchIndexConditions = null, PreparedStatementConditionBuilder &$additionalConditions = null) {
		if (!$this->findThreads) return '';
		
		if (empty($q)) {
			return "SELECT		thread.threadID AS objectID, thread.topic AS subject, thread.time, thread.username,
						'com.woltlab.wbb.post' AS objectType
				FROM		wbb".WCF_N."_thread thread
				INNER JOIN	(
							SELECT		DISTINCT post.threadID
							FROM		wbb".WCF_N."_post post
							WHERE		".$searchIndexConditions."
									AND post.isDeleted = 0
									AND post.isDisabled = 0
									".($this->findAttachments ? "AND post.attachments > 0" : '')."
									".($this->findPolls ? "AND post.pollID IS NOT NULL" : '')."
						) AS subselect
				ON		(thread.threadID = subselect.threadID)
				".($additionalConditions !== null ? $additionalConditions : '');
		}
		else {
			return "SELECT		thread.threadID AS objectID, thread.topic AS subject, thread.time, thread.username,
						'com.woltlab.wbb.post' AS objectType
				FROM		wbb".WCF_N."_thread thread
				INNER JOIN	(
							SELECT		DISTINCT post.threadID
							FROM		wbb".WCF_N."_post post
							INNER JOIN	(
										{WCF_SEARCH_INNER_JOIN}
									) search_index
							ON		(post.postID = search_index.objectID)
							WHERE		post.isDeleted = 0
									AND post.isDisabled = 0
									".($this->findAttachments ? "AND post.attachments > 0" : '')."
									".($this->findPolls ? "AND post.pollID IS NOT NULL" : '')."
						) AS subselect
				ON		(thread.threadID = subselect.threadID)
				".($additionalConditions !== null ? $additionalConditions : '');
		}
	}
	
	/**
	 * Reads the given form parameters.
	 */
	protected function readFormParameters() {
		$this->findThreads = 0;
		
		// get new values
		if (isset($_POST['boardIDs']) && is_array($_POST['boardIDs'])) {
			$this->boardIDs = ArrayUtil::toIntegerArray($_POST['boardIDs']);
		}
		
		if (isset($_POST['findAttachments'])) {
			$this->findAttachments = intval($_POST['findAttachments']);
		}
		
		if (isset($_POST['findPolls'])) {
			$this->findPolls = intval($_POST['findPolls']);
		}
		
		if (isset($_POST['findThreads'])) {
			$this->findThreads = intval($_POST['findThreads']);
		}
		
		if (isset($_REQUEST['findUserThreads'])) {
			$this->findUserThreads = intval($_REQUEST['findUserThreads']);
			if ($this->findUserThreads) $this->findThreads = 1;
		}
		
		if (isset($_POST['threadID'])) {
			$this->threadID = intval($_POST['threadID']);
			
			// make sure that the results are never grouped by thread
			$this->findThreads = 0;
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function setLocation() {
		WBBCore::getInstance()->setLocation();
	}
}
