<?php
namespace wbb\system\moderation\queue;
use wbb\data\board\BoardCache;
use wbb\data\post\Post;
use wbb\data\post\PostAction;
use wbb\data\post\PostList;
use wbb\system\cache\runtime\PostRuntimeCache;
use wbb\system\cache\runtime\ThreadRuntimeCache;
use wcf\data\moderation\queue\ModerationQueue;
use wcf\data\user\User;
use wcf\data\user\UserProfile;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\moderation\queue\AbstractModerationQueueHandler;
use wcf\system\moderation\queue\ModerationQueueManager;
use wcf\system\WCF;

/**
 * An abstract implementation of IModerationQueueHandler for forum posts.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Moderation\Queue
 */
abstract class AbstractPostModerationQueueHandler extends AbstractModerationQueueHandler {
	/**
	 * @inheritDoc
	 */
	protected $className = Post::class;
	
	/**
	 * @inheritDoc
	 */
	public function assignQueues(array $queues) {
		$assignments = $orphanedQueueIDs = $allPostIDs = [];
		foreach ($queues as $queue) {
			$allPostIDs[] = $queue->objectID;
		}
		
		$idsPerRun = 5000;
		$runs = ceil(count($allPostIDs) / $idsPerRun);
		for ($i = 0; $i < $runs; $i++) {
			$postIDs = array_slice($allPostIDs, $i * $idsPerRun, $idsPerRun);
			
			// fetch board id for each post
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add("post.postID IN (?)", [$postIDs]);
			$sql = "SELECT		post.postID, thread.boardID
				FROM		wbb" . WCF_N . "_post post
				LEFT JOIN	wbb" . WCF_N . "_thread thread
				ON		(thread.threadID = post.threadID)
				" . $conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
			$boardIDs = $statement->fetchMap('postID', 'boardID');
			
			foreach ($queues as $queue) {
				$assignUser = false;
				if (!isset($boardIDs[$queue->objectID])) {
					$orphanedQueueIDs[] = $queue->queueID;
					continue;
				}
				
				$board = BoardCache::getInstance()->getBoard($boardIDs[$queue->objectID]);
				if ($board->canEnter() && ($board->getModeratorPermission('canEnablePost') || $board->getModeratorPermission('canDeletePost'))) {
					$assignUser = true;
				}
				
				$assignments[$queue->queueID] = $assignUser;
			}
			
			ModerationQueueManager::getInstance()->removeOrphans($orphanedQueueIDs);
			ModerationQueueManager::getInstance()->setAssignment($assignments);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getContainerID($objectID) {
		$sql = "SELECT	boardID
			FROM	wbb".WCF_N."_thread
			WHERE	threadID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$this->getPost($objectID)->threadID]);
		$row = $statement->fetchArray();
		
		return $row === false ? 0 : $row['boardID'];
	}
	
	/**
	 * @inheritDoc
	 */
	public function isValid($objectID) {
		if ($this->getPost($objectID) === null) {
			return false;
		}
		
		return true;
	}
	
	/**
	 * Returns a post object by post id or null if post id is invalid.
	 * 
	 * @param	integer		$objectID
	 * @return	Post
	 */
	protected function getPost($objectID) {
		return PostRuntimeCache::getInstance()->getObject($objectID);
	}
	
	/**
	 * @inheritDoc
	 */
	public function populate(array $queues) {
		$threads = $objectIDs = [];
		foreach ($queues as $object) {
			$objectIDs[] = $object->objectID;
		}
		
		// fetch posts
		$postList = new PostList();
		$postList->sqlSelects .= "user_avatar.*, user_table.*";
		$postList->sqlJoins .= " LEFT JOIN wcf".WCF_N."_user user_table ON (user_table.userID = post.userID)";
		$postList->sqlJoins .= " LEFT JOIN wcf".WCF_N."_user_avatar user_avatar ON (user_avatar.avatarID = user_table.avatarID)";
		$postList->setObjectIDs($objectIDs);
		$postList->readObjects();
		$posts = $postList->getObjects();
		
		// fetch threads
		if (!empty($posts)) {
			$threadIDs = [];
			foreach ($posts as $post) {
				$threadIDs[] = $post->threadID;
			}
			
			$threads = ThreadRuntimeCache::getInstance()->getObjects($threadIDs);
		}
		
		foreach ($queues as $object) {
			if (isset($posts[$object->objectID])) {
				$post = $posts[$object->objectID];
				$post->setThread($threads[$post->threadID]);
				
				$object->setAffectedObject($post);
			}
			else {
				$object->setIsOrphaned();
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function canRemoveContent(ModerationQueue $queue) {
		if ($this->isValid($queue->objectID)) {
			return $this->getPost($queue->objectID)->getThread()->getBoard()->getModeratorPermission('canDeletePost');
		}
		
		return false;
	}
	
	/**
	 * @inheritDoc
	 */
	public function removeContent(ModerationQueue $queue, $message) {
		if ($this->isValid($queue->objectID) && !$this->getPost($queue->objectID)->isDeleted) {
			$postAction = new PostAction([$this->getPost($queue->objectID)], 'trash', ['reason' => $message]);
			$postAction->executeAction();
		}
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function isAffectedUser(ModerationQueue $queue, $userID) {
		// fetch user permissions
		$user = new UserProfile(new User($userID));
		if ($user->getPermission('mod.general.canUseModeration')) {
			return true;
		}
		
		// fetch board id
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("post.postID = ?", [$queue->objectID]);
		$sql = "SELECT		thread.boardID
			FROM		wbb".WCF_N."_post post
			LEFT JOIN	wbb".WCF_N."_thread thread
			ON		(thread.threadID = post.threadID)
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		$boardID = $statement->fetchColumn();
		if (in_array($userID, BoardCache::getInstance()->getUserModerators($boardID))) {
			return true;
		}
		if (count(array_intersect($user->getGroupIDs(), BoardCache::getInstance()->getGroupModerators($boardID)))) {
			return true;
		}
		
		return false;
	}
}
