<?php
namespace wbb\system\log\modification;
use wbb\data\modification\log\ViewablePostModificationLog;
use wbb\data\post\Post;
use wbb\data\post\PostList;
use wbb\system\user\notification\object\PostModerationUserNotificationObject;
use wcf\data\modification\log\ModificationLog;
use wcf\system\log\modification\AbstractExtendedModificationLogHandler;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\WCF;

/**
 * Handles post modification logs.
 *
 * @author      Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license     WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package     WoltLabSuite\Forum\System\Log\Modification
 */
class PostModificationLogHandler extends AbstractExtendedModificationLogHandler {
	/**
	 * @inheritDoc
	 */
	protected $objectTypeName = 'com.woltlab.wbb.post';
	
	/**
	 * Adds a log entry for post close.
	 *
	 * @param        Post $post
	 */
	public function close(Post $post) {
		$this->fireNotification($post, $this->add($post, 'close'));
	}
	
	/**
	 * Adds a log entry for post delete.
	 *
	 * @param        Post $post
	 */
	public function delete(Post $post) {
		$this->add($post, 'delete', [
			'time' => $post->time,
			'username' => $post->username
		]);
	}
	
	/**
	 * Adds a log entry for post disabling.
	 *
	 * @param        Post $post
	 */
	public function disable(Post $post) {
		$this->add($post, 'disable');
	}
	
	/**
	 * Adds a log entry for post edit.
	 *
	 * @param        Post $post
	 * @param        string $reason
	 */
	public function edit(Post $post, $reason = '') {
		$this->add($post, 'edit', ['reason' => $reason]);
	}
	
	/**
	 * Adds a log entry for post enabling.
	 *
	 * @param        Post $post
	 */
	public function enable(Post $post) {
		$this->fireNotification($post, $this->add($post, 'enable'));
	}
	
	/**
	 * Adds a log entry for post open.
	 *
	 * @param        Post $post
	 */
	public function open(Post $post) {
		$this->add($post, 'open');
	}
	
	/**
	 * Adds a log entry for post restore.
	 *
	 * @param        Post $post
	 */
	public function restore(Post $post) {
		$this->add($post, 'restore');
	}
	
	/**
	 * Adds a log entry for post soft-delete (trash).
	 *
	 * @param        Post $post
	 * @param        string $reason
	 */
	public function trash(Post $post, $reason = '') {
		$this->fireNotification($post, $this->add($post, 'trash', ['reason' => $reason]));
	}
	
	/**
	 * Adds a log entry for merging posts.
	 *
	 * @param        Post $post
	 * @param        integer $mergedPostCount
	 * @param        string[] $mergedPostAuthors
	 * @since        5.0
	 */
	public function merge(Post $post, $mergedPostCount, array $mergedPostAuthors) {
		$this->add($post, 'merge', [
			'mergedPostAuthors' => $mergedPostAuthors,
			'mergedPostCount' => $mergedPostCount
		]);
	}
	
	/**
	 * Adds a log entry for moving a post to another thread.
	 *
	 * @param        Post $post
	 * @since        5.0
	 */
	public function move(Post $post) {
		$this->fireNotification($post, $this->add($post, 'move', [
			'oldThreadID' => $post->threadID,
			'oldThreadTopic' => $post->getThread()->topic
		]));
	}
	
	/**
	 * Adds a post modification log entry.
	 *
	 * @param        Post $post
	 * @param        string $action
	 * @param        array $additionalData
	 * @return        ModificationLog
	 */
	public function add(Post $post, $action, array $additionalData = []) {
		return $this->createLog($action, $post->postID, $post->threadID, $additionalData);
	}
	
	/**
	 * Fires a moderation notification.
	 *
	 * @param        Post $post
	 * @param        ModificationLog $modificationLog
	 * @since        5.1
	 */
	protected function fireNotification(Post $post, ModificationLog $modificationLog) {
		if (!WBB_THREAD_ENABLE_MODERATION_NOTIFICATION) {
			return;
		}
		
		if ($post->userID == WCF::getUser()->userID) {
			return;
		}
		
		if (!$post->canRead() && !($post->getThread()->canRead() && $modificationLog->action == 'trash')) {
			return;
		}
		
		UserNotificationHandler::getInstance()->fireEvent('moderate', 'com.woltlab.wbb.moderation.post', new PostModerationUserNotificationObject($modificationLog), [
			$post->userID
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getAvailableActions() {
		return [
			'close',
			'delete',
			'disable',
			'edit',
			'enable',
			'merge',
			'move',
			'open',
			'restore',
			'trash'
		];
	}
	
	/**
	 * @inheritDoc
	 */
	public function processItems(array $items) {
		$postIDs = [];
		/** @var ModificationLog $item */
		foreach ($items as $item) {
			$postIDs[] = $item->objectID;
		}
		
		$postList = new PostList();
		$postList->setObjectIDs($postIDs);
		$postList->readObjects();
		$posts = $postList->getObjects();
		
		foreach ($items as &$item) {
			$item = new ViewablePostModificationLog($item);
			if (isset($posts[$item->objectID])) {
				$item->setPost($posts[$item->objectID]);
			}
		}
		
		return $items;
	}
}
