<?php
namespace wbb\system\event\listener;
use wbb\data\thread\ThreadAction;
use wbb\data\thread\ThreadList;
use wcf\data\article\Article;
use wcf\data\article\ArticleAction;
use wcf\data\article\ArticleList;
use wcf\system\event\listener\IParameterizedEventListener;

/**
 * Creates the article thread.
 *
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Event\Listener
 */
class ArticleActionThreadListener implements IParameterizedEventListener {
	/**
	 * list of thread ids per article id for deletion
	 * @var integer[][]
	 */
	private $threadIDs = [];
	
	/**
	 * Creates threads for the given article.
	 * 
	 * @param	Article		$article
	 */
	private function createArticleThread(Article $article) {
		(new ThreadAction([], 'rebuildArticleThreads', [
			'article' => $article,
			'articleContents' => $article->getArticleContents()
		]))->executeAction();
	}
	
	/**
	 * Creates threads for the given article.
	 *
	 * @param	Article		$article
	 * @param	string		$actionName
	 */
	private function toggleArticleThread(Article $article, $actionName) {
		$threadIDs = [];
		
		if ($actionName === 'delete') {
			if (isset($this->threadIDs[$article->articleID])) {
				$threadIDs = $this->threadIDs[$article->articleID];
			}
		}
		else {
			foreach ($article->getArticleContents() as $content) {
				/** @noinspection PhpUndefinedFieldInspection */
				if ($content->articleThreadID) {
					/** @noinspection PhpUndefinedFieldInspection */
					$threadIDs[] = $content->articleThreadID;
				}
			}
		}
		
		if (!empty($threadIDs)) {
			$threadList = new ThreadList();
			$threadList->getConditionBuilder()->add("threadID IN (?)", [$threadIDs]);
			$threadList->readObjects();
			$threads = $threadList->getObjects();
			
			if (!empty($threads)) {
				(new ThreadAction($threads, $actionName))->executeAction();
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function execute($eventObj, $className, $eventName, array &$parameters) {
		/** @var ArticleAction $eventObj */
		$actionName = $eventObj->getActionName();
		
		if ($eventName === 'initializeAction') {
			if ($actionName === 'delete') {
				$articles = [];
				if (!empty($eventObj->getObjects())) {
					$articles = $eventObj->getObjects();
				}
				else if (!empty($eventObj->getObjectIDs())) {
					// if only article ids are passed to the article
					// action, `ArticleAction::getObjects() at this
					// point will only return an empty array and
					// `ArticleAction::readObjects()` is protected,
					// thus we need to read the articles manually
					$articleList = new ArticleList();
					$articleList->setObjectIDs($eventObj->getObjectIDs());
					$articleList->readObjects();
					
					$articles = $articleList->getObjects();
				}
				
				// store thread ids, because these might not be available after deletion
				foreach ($articles as $article) {
					foreach ($article->getArticleContents() as $content) {
						/** @noinspection PhpUndefinedFieldInspection */
						$threadID = $content->articleThreadID;
						if ($threadID) {
							if (!isset($this->threadIDs[$article->articleID])) $this->threadIDs[$article->articleID] = [];
							$this->threadIDs[$article->articleID] = $threadID;
						}
					}
				}
			}
			
			return;
		}
		
		switch ($actionName) {
			case 'create':
				$returnValues = $eventObj->getReturnValues();
				if ($returnValues['returnValues']->publicationStatus == Article::PUBLISHED) {
					$this->createArticleThread($returnValues['returnValues']);
				}
				break;
				
			case 'update':
				foreach ($eventObj->getObjects() as $article) {
					if ($article->publicationStatus == Article::PUBLISHED || isset($eventObj->getParameters()['data']['publicationStatus']) && $eventObj->getParameters()['data']['publicationStatus'] == Article::PUBLISHED) {
						$article = new Article($article->articleID); // reload article obj to get updated time value
						$this->createArticleThread($article);
					}
				}
				break;
				
			case 'restore':
			case 'trash':
			case 'delete':
				foreach ($eventObj->getObjects() as $articleEditor) {
					$this->toggleArticleThread($articleEditor->getDecoratedObject(), $actionName);
				}
				break;
		}
	}
}
