<?php
namespace wbb\system\event\listener;
use wbb\data\board\BoardCache;
use wbb\data\post\PostAction;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadAction;
use wcf\data\AbstractDatabaseObjectAction;
use wcf\data\DatabaseObjectEditor;
use wcf\data\language\Language;
use wcf\system\event\listener\IParameterizedEventListener;
use wcf\system\html\input\HtmlInputProcessor;
use wcf\system\language\LanguageFactory;
use wcf\system\tagging\TagEngine;
use wcf\system\WCF;

/**
 * Abstract implementation of an event listener that handles creating, updating, and deleting the
 * (discussion) thread belonging to certain other object.
 * 
 * @author	Matthias Schmidt, Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Event\Listener
 * @since	5.2
 */
abstract class AbstractObjectActionThreadListener implements IParameterizedEventListener {
	/**
	 * @inheritDoc
	 */
	public function execute($eventObj, $className, $eventName, array &$parameters) {
		/** @var AbstractDatabaseObjectAction $eventObj */
		
		switch ($eventObj->getActionName()) {
			case 'triggerPublication':
				/** @var DatabaseObjectEditor $object */
				foreach ($eventObj->getObjects() as $object) {
					$editorClass = get_class($object);
					$baseClass = call_user_func([$editorClass, 'getBaseClass']);
					
					// reload object
					$object = new $editorClass(new $baseClass($object->getObjectID()));
					
					if ($object->{$this->getThreadIDProperty()}) {
						continue;
					}
					
					$boardID = $this->getBoardID($object);
					if ($boardID === null) {
						continue;
					}
					
					$board = BoardCache::getInstance()->getBoard($boardID);
					if ($board === null || !$board->isBoard()) {
						return;
					}
					
					$language = WCF::getLanguage();
					if ($object->languageID) {
						$language = LanguageFactory::getInstance()->getLanguage($object->languageID);
					}
					
					// create thread
					$htmlInputProcessor = new HtmlInputProcessor();
					$htmlInputProcessor->process($this->getMessage($object, $language), 'com.woltlab.wbb.post');
					$threadData = [
						'data' => [
							'boardID' => $board->boardID,
							'languageID' => $object->languageID,
							'topic' => $this->getSubject($object, $language),
							'time' => $object->time,
							'userID' => $object->userID,
							'username' => $object->username
						],
						'board' => $board,
						'postData' => [],
						'tags' => $this->getTags($object),
						'htmlInputProcessor' => $htmlInputProcessor
					];
					
					/** @var Thread $thread */
					$thread = (new ThreadAction([], 'create', $threadData))->executeAction()['returnValues'];
					
					// update object's thread id
					$object->update([
						$this->getThreadIDProperty() => $thread->threadID
					]);
					
					// mark thread as read
					(new ThreadAction([$thread], 'markAsRead'))->executeAction();
				}
				break;
				
			case 'update':
				/** @var DatabaseObjectEditor $object */
				foreach ($eventObj->getObjects() as $object) {
					if ($object->{$this->getThreadIDProperty()}) {
						$thread = new Thread($object->{$this->getThreadIDProperty()});
						$post = $thread->getFirstPost();
						
						$editorClass = get_class($object);
						$baseClass = call_user_func([$editorClass, 'getBaseClass']);
						
						// reload object
						$object = new $editorClass(new $baseClass($object->getObjectID()));
						
						$language = WCF::getLanguage();
						if ($object->languageID) {
							$language = LanguageFactory::getInstance()->getLanguage($object->languageID);
						}
						
						// update thread
						$threadAction = new ThreadAction([$thread], 'update', [
							'data' => [
								'tags' => $this->getTags($object),
								'languageID' => $object->languageID,
								'topic' => $this->getSubject($object, $language)
							]
						]);
						$threadAction->executeAction();
						
						// update first post
						$htmlInputProcessor = new HtmlInputProcessor();
						$htmlInputProcessor->process($this->getMessage($object, $language), 'com.woltlab.wbb.post');
						$postAction = new PostAction([$post], 'update', [
							'htmlInputProcessor' => $htmlInputProcessor
						]);
						$postAction->executeAction();
					}
				}
				break;
			
			case 'disable':
			case 'enable':
			case 'trash':
			case 'restore':
			case 'delete':
				$threadIDs = [];
				foreach ($eventObj->getObjects() as $object) {
					if ($object->{$this->getThreadIDProperty()}) {
						$threadIDs[] = $object->{$this->getThreadIDProperty()};
					}
				}
				
				if (!empty($threadIDs)) {
					(new ThreadAction($threadIDs, $eventObj->getActionName()))->executeAction();
				}
				
				break;
		}
	}
	
	/**
	 * Returns the id of the board the thread for the given object is created in or `null` if
	 * no thread should be created for the given object.
	 * 
	 * @param	DatabaseObjectEditor	$object
	 * @return	null|integer
	 */
	abstract protected function getBoardID(DatabaseObjectEditor $object);
	
	/**
	 * Returns the message of the thread for the given object in the given language.
	 * 
	 * @param	DatabaseObjectEditor	$object
	 * @param	Language		$language
	 * @return	string
	 */
	protected function getMessage(DatabaseObjectEditor $object, Language $language) {
		return $this->getText('message', $object, $language);
	}
	
	/**
	 * Returns the name of the taggable object type or `null` if the object does not support
	 * tagging.
	 * 
	 * @return	null|string
	 */
	protected function getTaggableObjectType() {
		return null;
	}
	
	/**
	 * Returns the subject of the thread for the given object in the given language.
	 * 
	 * @param	DatabaseObjectEditor	$object
	 * @param	Language		$language
	 * @return	string
	 */
	protected function getSubject(DatabaseObjectEditor $object, Language $language) {
		return $this->getText('subject', $object, $language);
	}
	
	/**
	 * Returns the titles of the tags of the given object.
	 * 
	 * @param	DatabaseObjectEditor	$object
	 * @return	string[]
	 */
	protected function getTags(DatabaseObjectEditor $object) {
		// get tags
		$tags = [];
		if (MODULE_TAGGING && $this->getTaggableObjectType()) {
			$tagObjects = TagEngine::getInstance()->getObjectTags(
				$this->getTaggableObjectType(),
				$object->getObjectID(),
				[$object->languageID === null ? LanguageFactory::getInstance()->getDefaultLanguageID() : '']
			);
			
			foreach ($tagObjects as $tagObject) {
				$tags[] = $tagObject->getTitle();
			}
		}
		
		return $tags;
	}
	
	/**
	 * Returns certain text for the thread for the given object in the given language.
	 * 
	 * @param	string			$text
	 * @param	DatabaseObjectEditor	$object
	 * @param	Language		$language
	 * @return	string
	 */
	protected function getText($text, DatabaseObjectEditor $object, Language $language) {
		$baseClass = call_user_func([$object, 'getBaseClass']);
		$classPieces = explode('\\', $baseClass);
		$application = $classPieces[0];
		$objectType = lcfirst(end($classPieces));
		
		return $language->getDynamicVariable(
			"{$application}.{$objectType}.{$objectType}Thread.{$text}",
			[$objectType => $object->getDecoratedObject()]
		);
	}
	
	/**
	 * Returns the name of the object property that stores the id of the associated thread in
	 * the relevant database objects.
	 * 
	 * @return	string
	 */
	abstract protected function getThreadIDProperty();
}
