<?php
namespace wbb\system\condition\thread;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadList;
use wbb\page\IThreadPage;
use wcf\data\condition\Condition;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\DatabaseObjectList;
use wcf\data\object\type\ObjectTypeCache;
use wcf\system\condition\AbstractSelectCondition;
use wcf\system\condition\IContentCondition;
use wcf\system\condition\IObjectCondition;
use wcf\system\condition\IObjectListCondition;
use wcf\system\request\RequestHandler;
use wcf\system\WCF;

/**
 * Condition implementation for the subscription state of a thread.
 * 
 * @author	Matthias Schmidt
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Condition\Thread
 * @since	5.2
 */
class ThreadSubscriptionCondition extends AbstractSelectCondition implements IContentCondition, IObjectCondition, IObjectListCondition {
	/**
	 * @inheritDoc
	 */
	protected $fieldName = 'wbbThreadSubscription';
	
	/**
	 * @inheritDoc
	 */
	protected $label = 'wbb.thread.condition.subscription';
	
	/**
	 * value of the "subscribed threads" option
	 * @var	integer
	 */
	const THREAD_SUBSCRIPTION = 1;
	
	/**
	 * value of the "subscribed boards" option
	 * @var	integer
	 */
	const BOARD_SUBSCRIPTION = 2;
	
	/**
	 * value of the "subscribed threads and subscribed boards" option
	 * @var	integer
	 */
	const BOARD_OR_THREAD_SUBSCRIPTION = 3;
	
	/**
	 * @inheritDoc
	 */
	public function addObjectListCondition(DatabaseObjectList $objectList, array $conditionData) {
		if (!($objectList instanceof ThreadList)) {
			throw new \InvalidArgumentException("Object list is no instance of '" . ThreadList::class . "', instance of '" . get_class($objectList) . "' given.");
		}
		
		// only registered users can subscribe to threads or boards
		if (!WCF::getUser()->userID) {
			$objectList->getConditionBuilder()->add('1 = 0');
		}
		else {
			$boardObjectTypeID = ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.user.objectWatch', 'com.woltlab.wbb.board');
			$threadObjectTypeID = ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.user.objectWatch', 'com.woltlab.wbb.thread');
			
			switch ($conditionData[$this->fieldName]) {
				case self::BOARD_SUBSCRIPTION:
					$objectList->getConditionBuilder()->add(
						"thread.boardID IN (
								SELECT	objectID
								FROM	wcf" . WCF_N . "_user_object_watch
								WHERE	objectTypeID = ?
									AND userID = ?
							)",
						[
							$boardObjectTypeID,
							WCF::getUser()->userID
						]
					);
					break;
					
				case self::BOARD_OR_THREAD_SUBSCRIPTION:
					$objectList->getConditionBuilder()->add(
						"(thread.boardID IN (
								SELECT	objectID
								FROM	wcf" . WCF_N . "_user_object_watch
								WHERE	objectTypeID = ?
									AND userID = ?
							)) OR (thread.threadID IN (
								SELECT	objectID
								FROM	wcf" . WCF_N . "_user_object_watch
								WHERE	objectTypeID = ?
									AND userID = ?
							))",
						[
							$boardObjectTypeID,
							WCF::getUser()->userID,
							$threadObjectTypeID,
							WCF::getUser()->userID
						]
					);
					break;
					
				case self::THREAD_SUBSCRIPTION:
					$objectList->getConditionBuilder()->add(
						"thread.threadID IN (
								SELECT	objectID
								FROM	wcf" . WCF_N . "_user_object_watch
								WHERE	objectTypeID = ?
									AND userID = ?
							)",
						[
							$threadObjectTypeID,
							WCF::getUser()->userID
						]
					);
					break;
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function checkObject(DatabaseObject $object, array $conditionData) {
		if (!($object instanceof Thread) && (!($object instanceof DatabaseObjectDecorator) || !($object->getDecoratedObject() instanceof Thread))) {
			throw new \InvalidArgumentException("Object is no (decorated) instance of '" . Thread::class . "', instance of '" . get_class($object) . "' given.");
		}
		
		switch ($conditionData[$this->fieldName]) {
			case self::BOARD_SUBSCRIPTION:
				return $object->getBoard()->isSubscribed();
			
			case self::BOARD_OR_THREAD_SUBSCRIPTION:
				return $object->isSubscribed() || $object->getBoard()->isSubscribed();
			
			case self::THREAD_SUBSCRIPTION:
				return $object->isSubscribed();
		}
		
		throw new \LogicException("Unreachable");
	}
	
	/**
	 * @inheritDoc
	 */
	public function showContent(Condition $condition) {
		$requestObject = RequestHandler::getInstance()->getActiveRequest()->getRequestObject();
		if (!($requestObject instanceof IThreadPage)) return false;
		
		return $this->checkObject($requestObject->getThread(), $condition->conditionData);
	}
	
	/**
	 * @inheritDoc
	 */
	protected function getOptions() {
		return [
			self::NO_SELECTION_VALUE => 'wcf.global.noSelection',
			self::THREAD_SUBSCRIPTION => 'wbb.thread.condition.subscription.threadSubscription',
			self::BOARD_SUBSCRIPTION => 'wbb.thread.condition.subscription.boardSubscription',
			self::BOARD_OR_THREAD_SUBSCRIPTION => 'wbb.thread.condition.subscription.boardOrThreadSubscription'
		];
	}
}
