<?php
namespace wbb\system\condition\thread;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadList;
use wbb\page\IThreadPage;
use wcf\data\condition\Condition;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\DatabaseObjectList;
use wcf\system\condition\AbstractSingleFieldCondition;
use wcf\system\condition\IContentCondition;
use wcf\system\condition\IObjectCondition;
use wcf\system\condition\IObjectListCondition;
use wcf\system\exception\UserInputException;
use wcf\system\request\RequestHandler;
use wcf\system\WCF;

/**
 * Condition implementation for the state of a thread.
 * 
 * @author	Matthias Schmidt
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Condition\Thread
 * @since	5.0
 */
class ThreadStateCondition extends AbstractSingleFieldCondition implements IContentCondition, IObjectCondition, IObjectListCondition {
	/**
	 * @inheritDoc
	 */
	protected $label = 'wbb.thread.condition.state';
	
	/**
	 * values of the possible state
	 * @var	integer[]
	 */
	public $states = [
		'isDeleted' => 0,
		'isNotDeleted' => 0,
		'isDisabled' => 0,
		'isEnabled' => 0,
		'isClosed' => 0,
		'isOpen' => 0,
		'isLink' => 0,
		'isNoLink' => 0,
		'isAnnouncement' => 0,
		'isSticky' => 0,
		'isNormal' => 0
	];
	
	/**
	 * names of states that are ignored if the condition works on `AccessibleThreadList` objects
	 * @var	string[]
	 */
	public $accessibleOnlyIgnoredStates = [
		'isDeleted',
		'isNotDeleted',
		'isDisabled',
		'isEnabled',
		'isLink',
		'isNoLink'
	];
	
	/**
	 * @inheritDoc
	 */
	public function __construct(DatabaseObject $object) {
		parent::__construct($object);
		
		if (WBB_MODULE_THREAD_MARKING_AS_DONE) {
			$this->states['isDone'] = 0;
			$this->states['isUndone'] = 0;
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function addObjectListCondition(DatabaseObjectList $objectList, array $conditionData) {
		if (!($objectList instanceof ThreadList)) {
			throw new \InvalidArgumentException("Object list is no instance of '".ThreadList::class."', instance of '".get_class($objectList)."' given.");
		}
		
		if (isset($conditionData['isDeleted'])) {
			$objectList->getConditionBuilder()->add('thread.isDeleted = ?', [$conditionData['isDeleted']]);
		}
		
		if (isset($conditionData['isDisabled'])) {
			$objectList->getConditionBuilder()->add('thread.isDisabled = ?', [$conditionData['isDisabled']]);
		}
		
		if (isset($conditionData['isClosed'])) {
			$objectList->getConditionBuilder()->add('thread.isClosed = ?', [$conditionData['isClosed']]);
		}
		
		if (isset($conditionData['isLink'])) {
			if ($conditionData['isLink']) {
				$objectList->getConditionBuilder()->add('thread.movedThreadID IS NOT NULL');
			}
			else {
				$objectList->getConditionBuilder()->add('thread.movedThreadID IS NULL');
			}
		}
		
		if (isset($conditionData['isAnnouncement'])) {
			$objectList->getConditionBuilder()->add('thread.isAnnouncement = ?', [1]);
		}
		if (isset($conditionData['isSticky'])) {
			$objectList->getConditionBuilder()->add('thread.isSticky = ?', [1]);
		}
		if (isset($conditionData['isNormal'])) {
			$objectList->getConditionBuilder()->add('thread.isAnnouncement = ?', [0]);
			$objectList->getConditionBuilder()->add('thread.isSticky = ?', [0]);
		}
		
		if (isset($conditionData['isDone'])) {
			$objectList->getConditionBuilder()->add('thread.isDone = ?', [$conditionData['isDone']]);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function checkObject(DatabaseObject $object, array $conditionData) {
		if (!($object instanceof Thread) && (!($object instanceof DatabaseObjectDecorator) || !($object->getDecoratedObject() instanceof Thread))) {
			throw new \InvalidArgumentException("Object is no (decorated) instance of '".Thread::class."', instance of '".get_class($object)."' given.");
		}
		
		$simpleStates = ['isDeleted', 'isDisabled', 'isClosed', 'isDone'];
		foreach ($simpleStates as $state) {
			/** @noinspection PhpVariableVariableInspection */
			if (isset($conditionData[$state]) && $object->$state != $conditionData[$state]) {
				return false;
			}
		}
		
		if (isset($conditionData['isLink'])) {
			if ($conditionData['isLink'] && $object->movedThreadID === null) {
				return false;
			}
			else if (!$conditionData['isLink'] && $object->movedThreadID !== null) {
				return false;
			}
		}
		
		if (isset($conditionData['isAnnouncement']) && !$object->isAnnouncement) {
			return false;
		}
		if (isset($conditionData['isSticky']) && !$object->isSticky) {
			return false;
		}
		if (isset($conditionData['isNormal']) && ($object->isAnnouncement || $object->isSticky)) {
			return false;
		}
		
		return true;
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getData() {
		$data = [];
		
		if ($this->states['isDeleted']) $data['isDeleted'] = 1;
		else if ($this->states['isNotDeleted']) $data['isDeleted'] = 0;
		
		if ($this->states['isDisabled']) $data['isDisabled'] = 1;
		else if ($this->states['isEnabled']) $data['isDisabled'] = 0;
		
		if ($this->states['isClosed']) $data['isClosed'] = 1;
		else if ($this->states['isOpen']) $data['isClosed'] = 0;
		
		if (!empty($this->states['isDone'])) $data['isDone'] = 1;
		else if (!empty($this->states['isUndone'])) $data['isDone'] = 0;
		
		if ($this->states['isLink']) $data['isLink'] = 1;
		else if ($this->states['isNoLink']) $data['isLink'] = 0;
		
		if ($this->states['isAnnouncement']) $data['isAnnouncement'] = 1;
		if ($this->states['isSticky']) $data['isSticky'] = 1;
		if ($this->states['isNormal']) $data['isNormal'] = 1;
		
		if ($this->object->accessibleOnly) {
			unset($data['isDeleted'], $data['isDisabled'], $data['isLink']);
		}
		
		if (!empty($data)) {
			return $data;
		}
		
		return null;
	}
	
	/**
	 * @inheritDoc
	 */
	protected function getFieldElement() {
		$fieldElement = '';
		
		foreach ($this->states as $state => $value) {
			if (!$this->object->accessibleOnly || !in_array($state, $this->accessibleOnlyIgnoredStates)) {
				$fieldElement .= '<label><input type="checkbox" name="wbbThread' . ucfirst($state) . '" value="1"' . ($value ? ' checked' : '') . '> ' . WCF::getLanguage()->get('wbb.thread.condition.state.' . $state) . "</label>\n";
			}
		}
		
		return $fieldElement;
	}
	
	/**
	 * @inheritDoc
	 */
	public function readFormParameters() {
		foreach ($this->states as $state => &$value) {
			if (isset($_POST['wbbThread'.ucfirst($state)])) $value = 1;
		}
		unset($value);
	}
	
	/**
	 * @inheritDoc
	 */
	public function reset() {
		foreach ($this->states as $state => $value) {
			$this->states[$state] = 0;
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function setData(Condition $condition) {
		/** @noinspection PhpUndefinedFieldInspection */
		$isDeleted = $condition->isDeleted;
		if ($isDeleted !== null) {
			$this->states['isDeleted'] = $isDeleted;
			$this->states['isNotDeleted'] = 1 - $isDeleted;
		}
		
		/** @noinspection PhpUndefinedFieldInspection */
		$isDisabled = $condition->isDisabled;
		if ($isDisabled !== null) {
			$this->states['isDisabled'] = $isDisabled;
			$this->states['isEnabled'] = 1 - $isDisabled;
		}
		
		/** @noinspection PhpUndefinedFieldInspection */
		$isClosed = $condition->isClosed;
		if ($isClosed !== null) {
			$this->states['isClosed'] = $isClosed;
			$this->states['isOpen'] = 1 - $isClosed;
		}
		
		/** @noinspection PhpUndefinedFieldInspection */
		$isDone = $condition->isDone;
		if ($isDone !== null) {
			$this->states['isDone'] = $isDone;
			$this->states['isUndone'] = 1 - $isDone;
		}
		
		/** @noinspection PhpUndefinedFieldInspection */
		$isLink = $condition->isLink;
		if ($isLink !== null) {
			$this->states['isLink'] = $isLink;
			$this->states['isNoLink'] = 1 - $isLink;
		}
		
		/** @noinspection PhpUndefinedFieldInspection */
		if ($condition->isAnnouncement !== null) $this->states['isAnnouncement'] = 1;
		
		/** @noinspection PhpUndefinedFieldInspection */
		if ($condition->isSticky !== null) $this->states['isSticky'] = 1;
		
		/** @noinspection PhpUndefinedFieldInspection */
		if ($condition->isNormal !== null) $this->states['isNormal'] = 1;
	}
	
	/**
	 * @inheritDoc
	 */
	public function showContent(Condition $condition) {
		$requestObject = RequestHandler::getInstance()->getActiveRequest()->getRequestObject();
		if (!($requestObject instanceof IThreadPage)) return false;
		
		return $this->checkObject($requestObject->getThread(), $condition->conditionData);
	}
	
	/**
	 * @inheritDoc
	 */
	public function validate() {
		if ($this->object->accessibleOnly) {
			if ($this->states['isDeleted'] && $this->states['isNotDeleted']) {
				$this->errorMessage = 'wbb.thread.condition.state.isDeleted.error.conflict';
				
				throw new UserInputException('isDeleted', 'conflict');
			}
			
			if ($this->states['isDisabled'] && $this->states['isEnabled']) {
				$this->errorMessage = 'wbb.thread.condition.state.isDisabled.error.conflict';
				
				throw new UserInputException('isDisabled', 'conflict');
			}
			
			if ($this->states['isClosed'] && $this->states['isOpen']) {
				$this->errorMessage = 'wbb.thread.condition.state.isClosed.error.conflict';
				
				throw new UserInputException('isClosed', 'conflict');
			}
			
			if ($this->states['isLink'] && $this->states['isNoLink']) {
				$this->errorMessage = 'wbb.thread.condition.state.isLink.error.conflict';
				
				throw new UserInputException('isLink', 'conflict');
			}
		}
		
		if ($this->states['isNormal'] && $this->states['isAnnouncement']) {
			$this->errorMessage = 'wbb.thread.condition.state.isNormal.error.conflict.isAnnouncement';
			
			throw new UserInputException('isAnnouncement', 'conflictIsAnnouncement');
		}
		if ($this->states['isNormal'] && $this->states['isSticky']) {
			$this->errorMessage = 'wbb.thread.condition.state.isNormal.error.conflict.isSticky';
			
			throw new UserInputException('isNormal', 'conflictIsSticky');
		}
		if ($this->states['isAnnouncement'] && $this->states['isSticky']) {
			$this->errorMessage = 'wbb.thread.condition.state.isAnnouncement.error.conflict.sticky';
			
			throw new UserInputException('isAnnouncement', 'conflict');
		}
		
		if (!empty($this->states['isDone']) && !empty($this->states['isUndone'])) {
			$this->errorMessage = 'wbb.thread.condition.state.isDone.error.conflict';
			
			throw new UserInputException('isDone', 'conflict');
		}
	}
}
