<?php
namespace wbb\system\condition\thread;
use wbb\data\board\BoardCache;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadList;
use wbb\page\IThreadPage;
use wbb\system\label\object\ThreadLabelObjectHandler;
use wcf\data\condition\Condition;
use wcf\data\label\group\ViewableLabelGroup;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\DatabaseObjectList;
use wcf\system\condition\AbstractCondition;
use wcf\system\condition\IContentCondition;
use wcf\system\condition\IObjectCondition;
use wcf\system\condition\IObjectListCondition;
use wcf\system\exception\UserInputException;
use wcf\system\label\LabelHandler;
use wcf\system\request\RequestHandler;
use wcf\system\WCF;

/**
 * Condition implementation for the labels of a thread.
 * 
 * Note: This condition only works in combination with ThreadBoardCondition.
 * 
 * @author	Matthias Schmidt
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Condition\Thread
 * @since	5.0
 */
class ThreadLabelCondition extends AbstractCondition implements IContentCondition, IObjectCondition, IObjectListCondition {
	/**
	 * id of the selector for the boards
	 * @var	string
	 */
	protected $boardSelectorID = 'wbbThreadBoardIDs';
	
	/**
	 * type of the thread label error
	 * @var	array
	 */
	protected $errorType = [];
	
	/**
	 * maps the label groups ids to the ids of the boards with whom they are
	 * associated
	 * @var	array
	 */
	public $labelGroupMapping = [];
	
	/**
	 * list of available label groups for all boards
	 * @var	ViewableLabelGroup[]
	 */
	public $labelGroups = [];
	
	/**
	 * list of selected label ids
	 * @var	integer[]
	 */
	public $labelIDs = [];
	
	/**
	 * @inheritDoc
	 */
	public function __construct(DatabaseObject $object) {
		parent::__construct($object);
		
		$this->labelGroups = ThreadLabelObjectHandler::getInstance()->getLabelGroups();
		$this->labelGroupMapping = BoardCache::getInstance()->getLabelGroupIDs();
		
		// `$this->labelGroupMapping` contains a mapping `boardID -> labelGroupIDs`,
		// while we need the opposite mapping for filtering out unused label groups
		$labelGroupIDToBoardIDs = [];
		foreach ($this->labelGroupMapping as $boardID => $labelGroupIDs) {
			foreach ($labelGroupIDs as $labelGroupID) {
				if (!isset($labelGroupIDToBoardIDs[$labelGroupID])) $labelGroupIDToBoardIDs[$labelGroupID] = [];
				$labelGroupIDToBoardIDs[$labelGroupID][] = $boardID;
			}
		}
		
		foreach ($this->labelGroups as $labelGroup) {
			if (!isset($labelGroupIDToBoardIDs[$labelGroup->groupID])) {
				unset($this->labelGroups[$labelGroup->groupID]);
			}
		}
		
		/** @noinspection PhpUndefinedFieldInspection */
		if ($this->object->boardSelectorID) {
			/** @noinspection PhpUndefinedFieldInspection */
			$this->boardSelectorID = $this->object->boardSelectorID;
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function addObjectListCondition(DatabaseObjectList $objectList, array $conditionData) {
		if (!($objectList instanceof ThreadList)) {
			throw new \InvalidArgumentException("Object list is no instance of '".ThreadList::class."', instance of '".get_class($objectList)."' given.");
		}
		
		$availableLabelGroupIDs = array_keys(LabelHandler::getInstance()->getLabelGroups(ThreadLabelObjectHandler::getInstance()->getLabelGroupIDs()));
		if (empty($availableLabelGroupIDs)) {
			return;
		}
		
		$objectType = LabelHandler::getInstance()->getObjectType('com.woltlab.wbb.thread');
		foreach ($conditionData['labelIDs'] as $labelGroupID => $labelID) {
			if (in_array($labelGroupID, $availableLabelGroupIDs)) {
				if ($labelID == -1) {
					$groupLabelIDs = LabelHandler::getInstance()->getLabelGroup($labelGroupID)->getLabelIDs();
					
					if (!empty($groupLabelIDs)) {
						$objectList->getConditionBuilder()->add('thread.threadID NOT IN (SELECT objectID FROM wcf'.WCF_N.'_label_object WHERE objectTypeID = ? AND labelID IN (?))', [
							$objectType->objectTypeID,
							$groupLabelIDs
						]);
					}
				}
				else {
					$objectList->getConditionBuilder()->add('thread.threadID IN (SELECT objectID FROM wcf'.WCF_N.'_label_object WHERE objectTypeID = ? AND labelID = ?)', [$objectType->objectTypeID, $labelID]);
				}
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function checkObject(DatabaseObject $object, array $conditionData) {
		if (!($object instanceof Thread) && (!($object instanceof DatabaseObjectDecorator) || !($object->getDecoratedObject() instanceof Thread))) {
			throw new \InvalidArgumentException("Object is no (decorated) instance of '".Thread::class."', instance of '".get_class($object)."' given.");
		}
		
		$labels = ThreadLabelObjectHandler::getInstance()->getAssignedLabels([$object->threadID], false);
		if (!isset($labels[$object->threadID])) return false;
		
		$labels = $labels[$object->threadID];
		$labelIDs = array_keys($labels);
		
		foreach ($conditionData['labelIDs'] as $labelGroupID => $labelID) {
			if ($labelID == -1) {
				$groupLabelIDs = LabelHandler::getInstance()->getLabelGroup($labelGroupID)->getLabelIDs();
				
				if (!empty(array_intersect($groupLabelIDs, $labelIDs))) {
					return false;
				}
			}
			else if (!isset($labels[$labelID])) {
				return false;
			}
		}
		
		return true;
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getData() {
		if (empty($this->labelIDs)) {
			return null;
		}
		
		return ['labelIDs' => $this->labelIDs];
	}
	
	/**
	 * @inheritDoc
	 */
	public function getHTML() {
		return WCF::getTPL()->fetch('threadLabelCondition', 'wbb', [
			'boardSelectorID' => $this->boardSelectorID,
			'errorType' => $this->errorType,
			'labelIDs' => $this->labelIDs,
			'labelGroupMapping' => $this->labelGroupMapping,
			'labelGroups' => $this->labelGroups
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function readFormParameters() {
		if (isset($_POST['labelIDs']) && is_array($_POST['labelIDs'])) $this->labelIDs = $_POST['labelIDs'];
	}
	
	/**
	 * @inheritDoc
	 */
	public function reset() {
		$this->labelIDs = [];
	}
	
	/**
	 * @inheritDoc
	 */
	public function setData(Condition $condition) {
		/** @noinspection PhpUndefinedFieldInspection */
		$this->labelIDs = $condition->labelIDs;
	}
	
	/**
	 * @inheritDoc
	 */
	public function showContent(Condition $condition) {
		$requestObject = RequestHandler::getInstance()->getActiveRequest()->getRequestObject();
		if (!($requestObject instanceof IThreadPage)) return false;
		
		return $this->checkObject($requestObject->getThread(), $condition->conditionData);
	}
	
	/**
	 * @inheritDoc
	 */
	public function validate() {
		// remove "without label" labelIDs as validateLabelIDs() cannot handle them
		$labelIDs = $this->labelIDs;
		foreach ($labelIDs as $labelGroupID => $labelID) {
			if ($labelID == -1 && isset($this->labelGroups[$labelGroupID])) {
				unset($labelIDs[$labelGroupID]);
			}
		}
		
		$this->errorType = ThreadLabelObjectHandler::getInstance()->validateLabelIDs($labelIDs, '', false);
		foreach ($this->errorType as $key => $errorType) {
			// we don't care for missing label groups
			if ($errorType !== 'missing') {
				throw new UserInputException('labelIDs');
			}
			else {
				unset($this->errorType[$key]);
			}
		}
	}
}
