<?php
namespace wbb\system\condition\thread;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadList;
use wbb\page\IThreadPage;
use wcf\data\condition\Condition;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\DatabaseObjectList;
use wcf\system\condition\AbstractIntegerCondition;
use wcf\system\condition\IContentCondition;
use wcf\system\condition\IObjectCondition;
use wcf\system\condition\IObjectListCondition;
use wcf\system\request\RequestHandler;
use wcf\system\WCF;

/**
 * Condition implementation for an integer property of a thread.
 * 
 * @author	Matthias Schmidt
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Condition\Thread
 * @since	5.0
 */
class ThreadIntegerPropertyCondition extends AbstractIntegerCondition implements IContentCondition, IObjectCondition, IObjectListCondition {
	/**
	 * name of the relevant object property
	 * @var	string
	 */
	protected $propertyName = '';
	
	/**
	 * @inheritDoc
	 */
	public function addObjectListCondition(DatabaseObjectList $objectList, array $conditionData) {
		if (!($objectList instanceof ThreadList)) {
			throw new \InvalidArgumentException("Object list is no instance of '".ThreadList::class."', instance of '".get_class($objectList)."' given.");
		}
		
		if (isset($conditionData['greaterThan'])) {
			$objectList->getConditionBuilder()->add('thread.'.$this->getPropertyName().' > ?', [$conditionData['greaterThan']]);
		}
		if (isset($conditionData['lessThan'])) {
			$objectList->getConditionBuilder()->add('thread.'.$this->getPropertyName().' < ?', [$conditionData['lessThan']]);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function checkObject(DatabaseObject $object, array $conditionData) {
		if (!($object instanceof Thread) && (!($object instanceof DatabaseObjectDecorator) || !($object->getDecoratedObject() instanceof Thread))) {
			throw new \InvalidArgumentException("Object is no (decorated) instance of '".Thread::class."', instance of '".get_class($object)."' given.");
		}
		
		if (isset($conditionData['greaterThan']) && $object->{$this->getPropertyName()} <= $conditionData['greaterThan']) {
			return false;
		}
		if (isset($conditionData['lessThan']) !== null && $object->{$this->getPropertyName()} >= $conditionData['lessThan']) {
			return false;
		}
		
		return true;
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	protected function getIdentifier() {
		return 'wbbThread'.ucfirst($this->getPropertyName());
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	protected function getLabel() {
		return WCF::getLanguage()->get('wbb.thread.condition.'.$this->getPropertyName());
	}
	
	/**
	 * Returns the name of the relevant object property.
	 * 
	 * @return	string
	 */
	protected function getPropertyName() {
		if (empty($this->propertyName)) {
			/** @noinspection PhpUndefinedFieldInspection */
			$propertyName = $this->getDecoratedObject()->propertyname;
			if ($propertyName) {
				$this->propertyName = $propertyName;
			}
			else {
				$objectTypePieces = explode('.', $this->getDecoratedObject()->objectType);
				$this->propertyName = end($objectTypePieces);
			}
		}
		
		return $this->propertyName;
	}
	
	/**
	 * @inheritDoc
	 */
	public function showContent(Condition $condition) {
		$requestObject = RequestHandler::getInstance()->getActiveRequest()->getRequestObject();
		if (!($requestObject instanceof IThreadPage)) return false;
		
		return $this->checkObject($requestObject->getThread(), $condition->conditionData);
	}
}
