<?php
namespace wbb\system\clipboard\action;
use wbb\data\board\BoardCache;
use wbb\data\board\BoardList;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadAction;
use wcf\data\clipboard\action\ClipboardAction;
use wcf\data\label\group\LabelGroup;
use wcf\system\clipboard\action\AbstractClipboardAction;
use wcf\system\clipboard\ClipboardHandler;
use wcf\system\label\LabelHandler;
use wcf\system\WCF;

/**
 * Prepares clipboard editor items for posts.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Clipboard\Action
 */
class ThreadClipboardAction extends AbstractClipboardAction {
	/**
	 * @inheritDoc
	 */
	protected $actionClassActions = ['close', 'delete', 'done', 'enable', 'disable', 'open', 'removeLink', 'restore', 'trash', 'undone'];
	
	/**
	 * list of applicable and accessible label groups
	 * @var	LabelGroup[]
	 */
	protected $labelGroups = [];
	
	/**
	 * @inheritDoc
	 */
	protected $supportedActions = ['assignLabel', 'close', 'copy', 'delete', 'done', 'enable', 'disable', 'merge', 'move', 'open', 'removeLink', 'restore', 'trash', 'undone'];
	
	/**
	 * list of active thread objects
	 * @var	Thread[]
	 */
	protected $threads = [];
	
	/**
	 * @inheritDoc
	 */
	public function execute(array $objects, ClipboardAction $action) {
		if (empty($this->threads)) {
			$this->threads = $this->loadBoards($objects);
		}
		
		$item = parent::execute($objects, $action);
		if ($item === null) {
			return null;
		}
		
		// handle actions
		switch ($action->actionName) {
			case 'assignLabel':
				WCF::getTPL()->assign([
					'labelGroups' => $this->labelGroups
				]);
				
				$item->addParameter('boardID', ClipboardHandler::getInstance()->getPageObjectID());
				$item->addParameter('template', WCF::getTPL()->fetch('threadAssignLabel', 'wbb'));
			break;
			
			case 'copy':
				$item->addInternalData('confirmMessage', WCF::getLanguage()->getDynamicVariable('wcf.clipboard.item.com.woltlab.wbb.thread.copy.confirmMessage', [
					'count' => $item->getCount()
				]));
				
				$item->addParameter('boardID', ClipboardHandler::getInstance()->getPageObjectID());
			break;
			
			case 'delete':
				$item->addInternalData('confirmMessage', WCF::getLanguage()->getDynamicVariable('wcf.clipboard.item.com.woltlab.wbb.thread.delete.confirmMessage', [
					'count' => $item->getCount()
				]));
			break;
			
			case 'merge':
				if ($item->getCount() < 2) {
					return null;
				}
			break;
			
			case 'removeLink':
				$item->addInternalData('confirmMessage', WCF::getLanguage()->getDynamicVariable('wcf.clipboard.item.com.woltlab.wbb.thread.removeLink.confirmMessage', [
					'count' => $item->getCount()
				]));
			break;
			
			case 'trash':
				$item->addInternalData('confirmMessage', WCF::getLanguage()->getDynamicVariable('wcf.clipboard.item.com.woltlab.wbb.thread.trash.confirmMessage', [
					'count' => $item->getCount()
				]));
				$item->addInternalData('template', WCF::getTPL()->fetch('threadDeleteReason', 'wbb'));
			break;
		}
		
		return $item;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getClassName() {
		return ThreadAction::class;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTypeName() {
		return 'com.woltlab.wbb.thread';
	}
	
	/**
	 * Returns the ids of the relevant threads which can marked as done.
	 * 
	 * @return	integer[]
	 */
	protected function validateDone() {
		return $this->__validateDone(true);
	}
	
	/**
	 * Returns the ids of the relevant threads which can marked as undone.
	 * 
	 * @return	integer[]
	 */
	protected function validateUndone() {
		return $this->__validateDone(false);
	}
	
	/**
	 * Validates parameters to mark threads as done/undone.
	 * 
	 * @param	boolean		$markAsDone
	 * @return	integer[]
	 */
	protected function __validateDone($markAsDone = true) {
		$threadIDs = [];
		
		if (WBB_MODULE_THREAD_MARKING_AS_DONE) {
			foreach ($this->threads as $thread) {
				if (!$thread->getBoard()->enableMarkingAsDone || ($markAsDone && $thread->isDone) || (!$markAsDone && !$thread->isDone)) {
					continue;
				}
				
				if ($thread->getBoard()->getModeratorPermission('canMarkAsDoneThread')) {
					$threadIDs[] = $thread->threadID;
				}
				else if ($thread->getBoard()->getPermission('canMarkAsDoneOwnThread') && ($thread->userID == WCF::getUser()->userID)) {
					$threadIDs[] = $thread->threadID;
				}
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads valid for moving and returns their ids.
	 * 
	 * @return	integer[]
	 */
	protected function validateMove() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if (!$thread->movedThreadID && $thread->getBoard()->getModeratorPermission('canMoveThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads available for label assignment.
	 * 
	 * @return	integer[]
	 */
	public function validateAssignLabel() {
		$threadIDs = [];
		
		$boardID = ClipboardHandler::getInstance()->getPageObjectID();
		$labelGroupIDs = BoardCache::getInstance()->getLabelGroupIDs($boardID);
		if (!empty($labelGroupIDs)) {
			$this->labelGroups = LabelHandler::getInstance()->getLabelGroups($labelGroupIDs, true, 'canSetLabel');
			if (!empty($this->labelGroups)) {
				foreach ($this->threads as $thread) {
					if (!$thread->movedThreadID && $thread->boardID == $boardID) {
						$threadIDs[] = $thread->threadID;
					}
				}
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads valid for closing and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateClose() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if (!$thread->movedThreadID && !$thread->isClosed && $thread->getBoard()->getModeratorPermission('canCloseThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads valid for copying and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateCopy() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if (!$thread->movedThreadID && $thread->getBoard()->getModeratorPermission('canCopyThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads valid for opening and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateOpen() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if (!$thread->movedThreadID && $thread->isClosed && $thread->getBoard()->getModeratorPermission('canCloseThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads valid for trashing and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateTrash() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if (!$thread->movedThreadID && !$thread->isDeleted && $thread->getBoard()->getModeratorPermission('canDeleteThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads valid for deleting and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateDelete() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if (!$thread->movedThreadID && $thread->isDeleted && $thread->getBoard()->getModeratorPermission('canDeleteThreadCompletely')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads valid for restoring and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateRestore() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if (!$thread->movedThreadID && $thread->isDeleted && $thread->getBoard()->getModeratorPermission('canRestoreThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads valid for enabling and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateEnable() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if (!$thread->movedThreadID && $thread->isDisabled && !$thread->isDeleted && $thread->getBoard()->getModeratorPermission('canEnableThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads valid for disabling and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateDisable() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if (!$thread->movedThreadID && !$thread->isDisabled && !$thread->isDeleted && $thread->getBoard()->getModeratorPermission('canEnableThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads representing a link to a moved thread and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateRemoveLink() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if ($thread->movedThreadID && $thread->getBoard()->getModeratorPermission('canMoveThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Validates threads for merging.
	 * 
	 * @return	integer[]
	 */
	public function validateMerge() {
		$threadIDs = [];
		
		foreach ($this->threads as $thread) {
			if ($thread->canRead() && $thread->getBoard()->getModeratorPermission('canMergeThread')) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		return $threadIDs;
	}
	
	/**
	 * Loads associated boards for given threads.
	 * 
	 * @param	Thread[]	$threads
	 * @return	Thread[]
	 */
	protected function loadBoards(array $threads) {
		// get board objects
		$boardIDs = [];
		foreach ($threads as $thread) {
			$boardIDs[] = $thread->boardID;
		}
		
		if (empty($boardIDs)) {
			return [];
		}
		
		$boardList = new BoardList();
		$boardList->setObjectIDs($boardIDs);
		$boardList->readObjects();
		
		foreach ($boardList as $board) {
			foreach ($threads as $thread) {
				if ($board->boardID == $thread->boardID) {
					$thread->setBoard($board);
				}
			}
		}
		
		return $threads;
	}
}
