<?php
namespace wbb\system\clipboard\action;
use wbb\data\board\BoardCache;
use wbb\data\board\ModeratorBoardNodeList;
use wbb\data\post\Post;
use wbb\data\post\PostAction;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadList;
use wbb\page\ThreadPage;
use wbb\system\cache\runtime\ThreadRuntimeCache;
use wcf\data\clipboard\action\ClipboardAction;
use wcf\system\clipboard\action\AbstractClipboardAction;
use wcf\system\clipboard\ClipboardHandler;
use wcf\system\WCF;

/**
 * Prepares clipboard editor items for posts.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Clipboard\Action
 */
class PostClipboardAction extends AbstractClipboardAction {
	/**
	 * @inheritDoc
	 */
	protected $actionClassActions = ['delete', 'disable', 'enable', 'moveToExistingThread', 'restore', 'trash'];
	
	/**
	 * list of posts
	 * @var	Post[]
	 */
	public $posts = [];
	
	/**
	 * @inheritDoc
	 */
	protected $supportedActions = ['copyToExistingThread', 'copyToNewThread', 'delete', 'disable', 'enable', 'merge', 'moveToExistingThread', 'moveToNewThread', 'trash', 'restore'];
	
	/**
	 * @inheritDoc
	 */
	public function execute(array $objects, ClipboardAction $action) {
		if (empty($this->posts)) {
			$this->posts = $this->loadThreads($objects);
		}
		
		$item = parent::execute($objects, $action);
		if ($item === null) {
			return null;
		}
		
		// handle actions
		switch ($action->actionName) {
			case 'copyToExistingThread':
				$item->addInternalData('confirmMessage', WCF::getLanguage()->getDynamicVariable('wcf.clipboard.item.com.woltlab.wbb.post.copyToExistingThread.confirmMessage', [
					'count' => $item->getCount()
				]));
				$item->addParameter('threadID', ClipboardHandler::getInstance()->getPageObjectID());
			break;
			
			case 'copyToNewThread':
				$boardNodeList = new ModeratorBoardNodeList();
				$boardNodeList->readNodeTree();
				WCF::getTPL()->assign([
					'boardID' => $this->getPreselectedBoardID(),
					'boardNodeList' => $boardNodeList->getNodeList(),
					'newThreadTopic' => $this->getCommonThreadTopic(),
				]);
				
				$item->addParameter('template', WCF::getTPL()->fetch('postCopyToNewThread', 'wbb'));
			break;
			
			case 'delete':
				$item->addInternalData('confirmMessage', WCF::getLanguage()->getDynamicVariable('wcf.clipboard.item.com.woltlab.wbb.post.deleteCompletely.confirmMessage', [
					'count' => $item->getCount()
				]));
			break;
			
			case 'merge':
				if ($item->getCount() < 2) {
					return null;
				}
			break;
			
			case 'moveToExistingThread':
				$item->addInternalData('confirmMessage', WCF::getLanguage()->getDynamicVariable('wcf.clipboard.item.com.woltlab.wbb.post.moveToExistingThread.confirmMessage', [
					'count' => $item->getCount()
				]));
				$item->addInternalData('parameters', [
					'threadID' => ClipboardHandler::getInstance()->getPageObjectID()
				]);
			break;
			
			case 'moveToNewThread':
				$boardNodeList = new ModeratorBoardNodeList();
				$boardNodeList->readNodeTree();
				WCF::getTPL()->assign([
					'boardID' => $this->getPreselectedBoardID(),
					'boardNodeList' => $boardNodeList->getNodeList(),
					'newThreadTopic' => $this->getCommonThreadTopic(),
				]);
				
				$item->addParameter('template', WCF::getTPL()->fetch('postMoveToNewThread', 'wbb'));
			break;
			
			case 'trash':
				$item->addInternalData('confirmMessage', WCF::getLanguage()->getDynamicVariable('wcf.clipboard.item.com.woltlab.wbb.post.trash.confirmMessage', [
					'count' => $item->getCount()
				]));
				$item->addInternalData('template', WCF::getTPL()->fetch('postDeleteReason', 'wbb'));
			break;
		}
		
		return $item;
	}
	
	/**
	 * Returns the common thread title of all posts if they belong to the same thread. If there
	 * is at least one post from a different thread, an empty string is returned.
	 * 
	 * @return	string
	 */
	protected function getCommonThreadTopic() {
		$threadIDs = array_unique(array_map(function(Post $post) {
			return $post->threadID;
		}, $this->posts));
		if (count($threadIDs) === 1) {
			return reset($this->posts)->getThread()->topic;
		}
		
		return '';
	}
	
	/**
	 * Returns the id of the board selected by default when moving/copying posts ids.
	 * 
	 * @return	int
	 */
	protected function getPreselectedBoardID() {
		$pageClasses = ClipboardHandler::getInstance()->getPageClasses();
		if ($pageClasses[0] === ThreadPage::class) {
			return ThreadRuntimeCache::getInstance()->getObject(ClipboardHandler::getInstance()->getPageObjectID())->boardID;
		}
		
		return ClipboardHandler::getInstance()->getPageObjectID();
	}
	
	/**
	 * @inheritDoc
	 */
	public function getClassName() {
		return PostAction::class;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTypeName() {
		return 'com.woltlab.wbb.post';
	}
	
	/**
	 * Returns the ids of the posts which can be moved to trash bin.
	 * 
	 * @return	integer[]
	 */
	protected function validateTrash() {
		// check if user may delete given posts
		$postIDs = [];
		foreach ($this->posts as $post) {
			if (!$post->isDeleted && $post->canDelete()) {
				$postIDs[] = $post->postID;
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Returns the ids of the posts which can be deleted.
	 * 
	 * @return	integer[]
	 */
	protected function validateDelete() {
		// check if user may delete given posts
		$postIDs = [];
		foreach ($this->posts as $post) {
			if ($post->isDeleted && $post->canDeleteCompletely()) {
				$postIDs[] = $post->postID;
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Returns the ids of the posts which can be restored.
	 * 
	 * @return	integer[]
	 */
	protected function validateRestore() {
		// check if user may delete given posts
		$postIDs = [];
		foreach ($this->posts as $post) {
			if ($post->isDeleted && $post->canRestore()) {
				$postIDs[] = $post->postID;
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Validates posts valid for enabling and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateEnable() {
		$postIDs = [];
		
		foreach ($this->posts as $post) {
			if ($post->isDisabled && !$post->isDeleted && $post->canEnable()) {
				$postIDs[] = $post->postID;
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Validates posts valid for disabling and returns their ids.
	 * 
	 * @return	integer[]
	 */
	public function validateDisable() {
		$postIDs = [];
		
		foreach ($this->posts as $post) {
			if (!$post->isDisabled && !$post->isDeleted && $post->canEnable()) {
				$postIDs[] = $post->postID;
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Validates posts for merging.
	 * 
	 * @return	integer[]
	 */
	public function validateMerge() {
		$postIDs = [];
		foreach ($this->posts as $post) {
			if ($post->getThread()->canRead() && $post->getThread()->getBoard()->getModeratorPermission('canMergePost')) {
				$postIDs[] = $post->postID;
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Validates posts to move them into an existing thread.
	 * 
	 * @return	integer[]
	 */
	public function validateMoveToExistingThread() {
		$postIDs = $this->__validateMove();
		
		if (!empty($postIDs)) {
			// validate permissions for target thread
			$thread = new Thread(ClipboardHandler::getInstance()->getPageObjectID());
			if (!$thread->threadID) {
				$postIDs = [];
			}
			else {
				foreach ($postIDs as $key => $postID) {
					// cannot move into the same thread
					if ($this->posts[$postID]->threadID == $thread->threadID) {
						unset($postIDs[$key]);
					}
				}
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Validates posts to move them into a new thread.
	 * 
	 * @return	integer[]
	 */
	public function validateMoveToNewThread() {
		$postIDs = $this->__validateMove();
		
		if (!empty($postIDs)) {
			$board = null;
			
			$pageClasses = ClipboardHandler::getInstance()->getPageClasses();
			if ($pageClasses[0] === ThreadPage::class) {
				$thread = new Thread(ClipboardHandler::getInstance()->getPageObjectID());
				if ($thread->threadID) $board = $thread->getBoard();
			}
			else {
				$board = BoardCache::getInstance()->getBoard(ClipboardHandler::getInstance()->getPageObjectID());
			}
			
			// validate permissions for target board
			if ($board === null || !$board->canStartThread()) {
				$postIDs = [];
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Validates posts for moving.
	 * 
	 * @return	integer[]
	 */
	protected function __validateMove() {
		if (!ClipboardHandler::getInstance()->getPageObjectID()) {
			return [];
		}
		
		$postIDs = [];
		foreach ($this->posts as $post) {
			if ($post->getThread()->canRead() && $post->getThread()->getBoard()->getModeratorPermission('canMovePost')) {
				$postIDs[] = $post->postID;
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Validates posts to copy them into an existing thread.
	 * 
	 * @return	integer[]
	 */
	public function validateCopyToExistingThread() {
		$postIDs = $this->__validateCopy();
		
		if (!empty($postIDs)) {
			// validate permissions for target thread
			$thread = new Thread(ClipboardHandler::getInstance()->getPageObjectID());
			if (!$thread->threadID) {
				$postIDs = [];
			}
			else {
				foreach ($postIDs as $key => $postID) {
					// cannot copy into the same thread
					if ($this->posts[$postID]->threadID == $thread->threadID) {
						unset($postIDs[$key]);
					}
				}
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Validates posts to copy them into a new thread.
	 * 
	 * @return	integer[]
	 */
	public function validateCopyToNewThread() {
		$postIDs = $this->__validateCopy();
		
		if (!empty($postIDs)) {
			$board = null;
			
			$pageClasses = ClipboardHandler::getInstance()->getPageClasses();
			if ($pageClasses[0] === ThreadPage::class) {
				$thread = new Thread(ClipboardHandler::getInstance()->getPageObjectID());
				if ($thread->threadID) $board = $thread->getBoard();
			}
			else {
				$board = BoardCache::getInstance()->getBoard(ClipboardHandler::getInstance()->getPageObjectID());
			}
			
			// validate permissions for target board
			if ($board === null || !$board->canStartThread()) {
				$postIDs = [];
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Validates posts for copying.
	 * 
	 * @return	integer[]
	 */
	protected function __validateCopy() {
		if (!ClipboardHandler::getInstance()->getPageObjectID()) {
			return [];
		}
		
		$postIDs = [];
		foreach ($this->posts as $post) {
			if ($post->getThread()->canRead() && $post->getThread()->getBoard()->getModeratorPermission('canCopyPost')) {
				$postIDs[] = $post->postID;
			}
		}
		
		return $postIDs;
	}
	
	/**
	 * Loads threads for given posts.
	 * 
	 * @param	Post[]		$objects
	 * @return	Post[]
	 */
	protected function loadThreads(array $objects) {
		// get thread objects
		$threadIDs = [];
		foreach ($objects as $post) {
			$threadIDs[] = $post->threadID;
		}
		
		$threadList = new ThreadList();
		$threadList->setObjectIDs($threadIDs);
		$threadList->readObjects();
		
		foreach ($threadList as $thread) {
			foreach ($objects as $post) {
				if ($post->threadID == $thread->threadID) {
					$post->setThread($thread);
				}
			}
		}
		
		return $objects;
	}
}
