<?php
namespace wbb\system\cache\builder;
use wbb\data\board\BoardList;
use wcf\data\object\type\ObjectTypeCache;
use wcf\system\acl\ACLHandler;
use wcf\system\cache\builder\AbstractCacheBuilder;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\WCF;

/**
 * Caches all boards, the structure of boards and all moderators.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Cache\Builder
 */
class BoardCacheBuilder extends AbstractCacheBuilder {
	/**
	 * @inheritDoc
	 */
	protected function rebuild(array $parameters) {
		$data = [
			'boards' => [],
			'boardStructure' => [],
			'labelGroups' => [],
			'moderators' => [
				'users' => [],
				'groups' => []
			]
		];
		
		// boards
		$boardList = new BoardList();
		$boardList->sqlOrderBy = 'board.parentID ASC, board.position ASC';
		$boardList->readObjects();
		$data['boards'] = $boardList->getObjects();
		
		// board structure
		foreach ($boardList->getObjects() as $board) {
			$data['boardStructure'][$board->parentID][] = $board->boardID;
		}
		
		// label group associations
		$this->loadLabelGroups($data);
		
		// moderators
		$this->loadModerators($data);
		return $data;
	}
	
	/**
	 * Loads label group associations.
	 * 
	 * @param	integer[][][]	$data
	 */
	protected function loadLabelGroups(array &$data) {
		$boardIDs = array_keys($data['boards']);
		if (empty($boardIDs)) {
			return;
		}
		
		// get object type
		$objectType = ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.label.objectType', 'com.woltlab.wbb.board');
		if ($objectType === null) {
			return;
		}
		
		// fetch data
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("objectTypeID = ?", [$objectType->objectTypeID]);
		$conditions->add("objectID IN (?)", [$boardIDs]);
		
		$sql = "SELECT	groupID, objectID
			FROM	wcf".WCF_N."_label_group_to_object
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		while ($row = $statement->fetchArray()) {
			if (!isset($data['labelGroups'][$row['objectID']])) {
				$data['labelGroups'][$row['objectID']] = [];
			}
			
			$data['labelGroups'][$row['objectID']][] = $row['groupID'];
		}
	}
	
	/**
	 * Loads moderators.
	 * 
	 * @param	mixed[][]	$data
	 */
	protected function loadModerators(array &$data) {
		// users
		// read all user ids with at least one moderative permission granted / denied
		$sql = "SELECT		acl_option_to_user.userID, acl_option.optionName, acl_option_to_user.optionValue, acl_option_to_user.objectID
			FROM		wcf".WCF_N."_acl_option acl_option,
					wcf".WCF_N."_acl_option_to_user acl_option_to_user
			WHERE		acl_option_to_user.optionID = acl_option.optionID
					AND acl_option.objectTypeID = ?
					AND acl_option.categoryName LIKE ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([
			ACLHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board'),
			'mod.%'
		]);
		$moderators = [];
		while ($row = $statement->fetchArray()) {
			if (!isset($moderators[$row['objectID']])) {
				$moderators[$row['objectID']] = [];
			}
			if (!isset($moderators[$row['objectID']][$row['userID']])) {
				$moderators[$row['objectID']][$row['userID']] = [];
			}
			
			$moderators[$row['objectID']][$row['userID']][$row['optionName']] = $row['optionValue'];
		}
		$this->inheritModerators($data, $moderators, null, 'users');

		// groups
		// read all group ids with at least one moderative permission granted / denied
		$sql = "SELECT		acl_option_to_group.groupID, acl_option.optionName, acl_option_to_group.optionValue, acl_option_to_group.objectID
			FROM		wcf".WCF_N."_acl_option acl_option,
					wcf".WCF_N."_acl_option_to_group acl_option_to_group
			WHERE		acl_option_to_group.optionID = acl_option.optionID
					AND acl_option.objectTypeID = ?
					AND acl_option.categoryName LIKE ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([
			ACLHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board'),
			'mod.%'
		]);
		$moderators = [];
		while ($row = $statement->fetchArray()) {
			if (!isset($moderators[$row['objectID']])) {
				$moderators[$row['objectID']] = [];
			}
			if (!isset($moderators[$row['objectID']][$row['groupID']])) {
				$moderators[$row['objectID']][$row['groupID']] = [];
			}
			
			$moderators[$row['objectID']][$row['groupID']][$row['optionName']] = $row['optionValue'];
		}
		$this->inheritModerators($data, $moderators, null, 'groups');
		
		// remove duplicate values
		foreach ($data['moderators']['users'] as &$userIDs) {
			$userIDs = array_unique($userIDs, SORT_NUMERIC);
		}
		unset($userIDs);
		foreach ($data['moderators']['groups'] as &$groupIDs) {
			$groupIDs = array_unique($groupIDs, SORT_NUMERIC);
		}
		unset($groupIDs);
	}

	/**
	 * Inherits the moderators list down the board structure.
	 * 
	 * @param	array	$data		The array that will be returned by the cache builder.
	 * @param	array	$moderators	Array with the (inherited) permissions that are in effect in the board.
	 * @param	integer	$boardID	Current board
	 * @param	string	$type		Either 'users' or 'groups'.
	 */
	protected function inheritModerators(array &$data, array &$moderators, $boardID, $type = 'users') {
		if (isset($data['boardStructure'][$boardID])) {
			foreach ($data['boardStructure'][$boardID] as $childBoardID) {
				// Check whether the current board has permissions set (otherwise we cannot inherit).
				if (isset($moderators[$boardID])) {
					foreach ($moderators[$boardID] as $objectID => $optionNames) {
						foreach ($optionNames as $optionName => $optionValue) {
							// The current permission is not explicitely set in the child: inherit.
							if (!isset($moderators[$childBoardID][$objectID][$optionName])) {
								$moderators[$childBoardID][$objectID][$optionName] = $optionValue;
							}
						}
					}
				}
				
				$this->inheritModerators($data, $moderators, $childBoardID, $type);
			}
		}
		
		// Store the users / groups with at least one granted permission for each board.
		if (isset($moderators[$boardID])) {
			foreach ($moderators[$boardID] as $objectID => $optionNames) {
				if (count(array_filter($optionNames))) {
					$data['moderators'][$type][$boardID][] = $objectID;
				}
			}
		}
	}
}
