<?php
namespace wbb\system\bulk\processing\thread;
use wbb\data\thread\ThreadAction;
use wbb\data\thread\ThreadList;
use wcf\data\language\Language;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectList;
use wcf\system\exception\UserInputException;
use wcf\system\language\LanguageFactory;
use wcf\system\WCF;

/**
 * Bulk processing action implementation for changing the language of threads.
 * 
 * @author	Matthias Schmidt
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Bulk\Processing\Thread
 * @since	5.0
 */
class ChangeLanguageThreadBulkProcessingAction extends AbstractThreadBulkProcessingAction {
	/**
	 * list of available languages
	 * @var	Language[]
	 */
	public $languages = [];
	
	/**
	 * id of the new thread language
	 * @var	integer
	 */
	public $newLanguageID = 0;
	
	/**
	 * @inheritDoc
	 */
	public function __construct(DatabaseObject $object) {
		parent::__construct($object);
		
		if (LanguageFactory::getInstance()->multilingualismEnabled()) {
			$this->languages = LanguageFactory::getInstance()->getContentLanguages();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function executeAction(DatabaseObjectList $objectList) {
		if (!($objectList instanceof ThreadList)) {
			throw new \InvalidArgumentException("Object list is no instance of '".ThreadList::class."', instance of '".get_class($objectList)."' given.");
		}
		
		/** @noinspection PhpUndefinedFieldInspection */
		if ($this->object->action === null) return;
		
		if (count($objectList)) {
			$threadAction = new ThreadAction($objectList->getObjects(), 'update', [
				'data' => [
					'languageID' => $this->newLanguageID ?: null
				],
				'isBulkProcessing' => true
			]);
			$threadAction->executeAction();
		}
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getHTML() {
		return WCF::getTPL()->fetch('changeLanguageThreadBulkProcessing', 'wbb', [
			'newLanguages' => $this->languages,
			'newLanguageID' => $this->newLanguageID
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function readFormParameters() {
		if (isset($_POST['newLanguageID'])) $this->newLanguageID = intval($_POST['newLanguageID']);
	}
	
	/**
	 * @inheritDoc
	 */
	public function reset() {
		$this->newLanguageID = 0;
	}
	
	/**
	 * @inheritDoc
	 */
	public function validate() {
		if ($this->newLanguageID && !isset($this->languages[$this->newLanguageID])) {
			throw new UserInputException('newLanguageID', 'noValidSelection');
		}
	}
}
