<?php
namespace wbb\system\box;
use wbb\data\thread\AccessibleThreadList;
use wbb\data\thread\CacheBuilderThreadList;
use wbb\data\thread\HotThreadList;
use wbb\system\cache\builder\HotThreadsCacheBuilder;
use wbb\system\cache\builder\LatestPostsCacheBuilder;
use wbb\system\cache\builder\MostActiveThreadsCacheBuilder;
use wcf\system\box\AbstractDatabaseObjectListBoxController;
use wcf\system\user\UserProfileHandler;
use wcf\system\WCF;

/**
 * Dynamic box controller implementation for a list of threads.
 *
 * @author	Matthias Schmidt
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Box
 * @since	5.0
 */
class ThreadListBoxController extends AbstractDatabaseObjectListBoxController {
	/**
	 * maps special sort fields to cache builders
	 * @var	string[]
	 */
	public $cacheBuilders = [
		'activity' => MostActiveThreadsCacheBuilder::class,
		'hotness' => HotThreadsCacheBuilder::class,
		'lastPostTime' => LatestPostsCacheBuilder::class
	];
	
	/**
	 * @inheritDoc
	 */
	protected $conditionDefinition = 'com.woltlab.wbb.box.threadList.condition';
	
	/**
	 * @inheritDoc
	 */
	public $defaultLimit = 5;
	
	/**
	 * @var	string[][]
	 */
	public $realSortFields = [
		'activity' => ['thread.replies', 'thread.lastPostTime'],
		'hotness' => [HotThreadList::TEMPERATURE_FORMULA, 'thread.time'],
		'lastPostTime' => ['thread.lastPostTime', 'thread.threadID']
	];
	
	/**
	 * @inheritDoc
	 */
	protected $sortFieldLanguageItemPrefix = 'wbb.thread.sortField';
	
	/**
	 * @inheritDoc
	 */
	protected static $supportedPositions = ['contentBottom', 'contentTop', 'sidebarLeft', 'sidebarRight'];
	
	/**
	 * @inheritDoc
	 */
	public $validSortFields = [
		'topic',
		'username',
		'time',
		'replies',
		'views',
		'lastPostTime',
		'hotness', // pseudo-sort field
		'activity' // pseudo-sort field
	];
	
	/**
	 * @inheritDoc
	 */
	public function __construct() {
		if (!empty($this->validSortFields) && MODULE_LIKE) {
			$this->validSortFields[] = 'cumulativeLikes';
		}
		
		parent::__construct();
	}
	
	/**
	 * @inheritDoc
	 */
	protected function getObjectList() {
		/** @noinspection PhpUndefinedFieldInspection */
		if ($this->box->sortOrder && $this->box->sortField && isset($this->cacheBuilders[$this->box->sortField])) {
			/** @noinspection PhpUndefinedFieldInspection */
			return new CacheBuilderThreadList(call_user_func([$this->cacheBuilders[$this->box->sortField], 'getInstance']));
		}
		
		return new AccessibleThreadList();
	}
	
	/**
	 * @inheritDoc
	 */
	protected function getTemplate() {
		$templateName = 'boxThreadListTable';
		if ($this->box->position === 'sidebarLeft' || $this->box->position === 'sidebarRight') {
			$templateName = 'boxThreadListSidebar';
		}
		else {
			// handle special display mode of non-sidebar boxes
			foreach ($this->box->getConditions() as $condition) {
				/** @noinspection PhpUndefinedFieldInspection */
				if ($condition->getObjectType()->objectType === 'com.woltlab.wbb.displayMode' && $condition->wbbThreadDisplayMode === 'firstPostExcerpt') {
					$templateName = 'boxThreadList';
					break;
				}
			}
		}
		
		return WCF::getTPL()->fetch($templateName, 'wbb', [
			'boxThreadList' => $this->objectList,
			'boxSortField' => $this->sortField
		], true);
	}
	
	/**
	 * @inheritDoc
	 */
	protected function readObjects() {
		/** @noinspection PhpUndefinedFieldInspection */
		if ($this->box->sortOrder && $this->box->sortField && isset($this->realSortFields[$this->box->sortField])) {
			/** @noinspection PhpUndefinedFieldInspection */
			$this->objectList->sqlOrderBy = implode(' '.$this->box->sortOrder.',', $this->realSortFields[$this->box->sortField]).' '.$this->box->sortOrder;
		}
		
		if (!empty(UserProfileHandler::getInstance()->getIgnoredUsers())) {
			$this->objectList->getConditionBuilder()->add("(thread.userID IS NULL OR thread.userID NOT IN (?))", [UserProfileHandler::getInstance()->getIgnoredUsers()]);
		}
		
		parent::readObjects();
	}
}
