<?php
namespace wbb\system\board;
use wbb\data\board\Board;
use wbb\system\cache\builder\BoardModeratorPermissionCacheBuilder;
use wbb\system\cache\builder\BoardPermissionCacheBuilder;
use wcf\data\user\User;
use wcf\system\acl\ACLHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\SingletonFactory;
use wcf\system\WCF;
use wcf\util\StringUtil;

/**
 * Caches board and moderator permissions.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\System\Board
 */
class BoardPermissionCache extends SingletonFactory {
	/**
	 * list of permissions
	 * @var	array
	 */
	protected $boardPermissions = [];
	
	/**
	 * list of mod permissions
	 * @var	array
	 */
	protected $moderatorPermissions = [];
	
	/**
	 * Returns the permissions for given user.
	 * 
	 * @param	User	$user
	 */
	protected function loadPermissions(User $user) {
		// get groups permissions
		$this->boardPermissions[$user->userID] = BoardPermissionCacheBuilder::getInstance()->getData($user->getGroupIDs());
		$this->moderatorPermissions[$user->userID] = BoardModeratorPermissionCacheBuilder::getInstance()->getData($user->getGroupIDs());
		
		// get user permissions
		if ($user->userID) {
			$data = UserStorageHandler::getInstance()->getField('wbbBoardPermissions', $user->userID);
			
			// cache does not exist or is outdated
			if ($data === null) {
				$moderatorPermissions = $userPermissions = [];
				
				$sql = "SELECT	option_to_user.objectID AS boardID, option_to_user.optionValue,
						acl_option.optionName AS permission, acl_option.categoryName
					FROM	wcf".WCF_N."_acl_option acl_option,
						wcf".WCF_N."_acl_option_to_user option_to_user
					WHERE	acl_option.objectTypeID = ?
						AND option_to_user.optionID = acl_option.optionID
						AND option_to_user.userID = ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute([
					ACLHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board'),
					$user->userID
				]);
				while ($row = $statement->fetchArray()) {
					if (StringUtil::startsWith($row['categoryName'], 'user.')) {
						$userPermissions[$row['boardID']][$row['permission']] = $row['optionValue'];
					}
					else {
						$moderatorPermissions[$row['boardID']][$row['permission']] = $row['optionValue'];
					}
				}
				
				if (!empty($userPermissions)) {
					Board::inheritPermissions(null, $userPermissions);
				}
				if (!empty($moderatorPermissions)) {
					Board::inheritPermissions(null, $moderatorPermissions);
				}
				
				// update storage data
				UserStorageHandler::getInstance()->update($user->userID, 'wbbBoardPermissions', serialize([
					'mod' => $moderatorPermissions,
					'user' => $userPermissions
				]));
			}
			else {
				$tmp = unserialize($data);
				$moderatorPermissions = $tmp['mod'];
				$userPermissions = $tmp['user'];
			}
			
			foreach ($userPermissions as $boardID => $permissions) {
				foreach ($permissions as $name => $value) {
					$this->boardPermissions[$user->userID][$boardID][$name] = $value;
				}
			}
			
			foreach ($moderatorPermissions as $boardID => $permissions) {
				foreach ($permissions as $name => $value) {
					$this->moderatorPermissions[$user->userID][$boardID][$name] = $value;
				}
			}
		}
	}
	
	/**
	 * Returns the board permissions for given user.
	 * 
	 * @param	User	$user
	 * @return	array
	 */
	public function getPermissions(User $user) {
		if (!isset($this->boardPermissions[$user->userID])) {
			$this->loadPermissions($user);
		}
		
		return $this->boardPermissions[$user->userID];
	}
	
	/**
	 * Returns the moderator permissions for given user.
	 * 
	 * @param	User	$user
	 * @return	array
	 */
	public function getModeratorPermissions(User $user) {
		if (!isset($this->moderatorPermissions[$user->userID])) {
			$this->loadPermissions($user);
		}
		
		return $this->moderatorPermissions[$user->userID];
	}
}
