<?php
namespace wbb\system\article\discussion;
use wbb\data\post\ViewablePostList;
use wbb\data\thread\Thread;
use wbb\data\thread\ViewableThread;
use wbb\system\cache\runtime\ThreadRuntimeCache;
use wcf\data\article\Article;
use wcf\system\article\discussion\AbstractArticleDiscussionProvider;
use wcf\system\reaction\ReactionHandler;
use wcf\system\WCF;

/**
 * A discussion provider that uses forum threads for article discussions.
 *
 * @author      Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license     WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package     WoltLabSuite\Forum\System\Article\Discussion
 * @since       5.2
 */
class ThreadArticleDiscussionProvider extends AbstractArticleDiscussionProvider {
	/**
	 * @inheritDoc
	 */
	public function __construct(Article $article) {
		parent::__construct($article);
		
		/** @noinspection PhpUndefinedFieldInspection */
		ThreadRuntimeCache::getInstance()->cacheObjectID($article->getArticleContent()->articleThreadID);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getDiscussionCount() {
		return $this->getThread() ? $this->getThread()->replies : 0;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getDiscussionLink() {
		return $this->getThread() ? $this->getThread()->getLink() : '';
	}
	
	/**
	 * @inheritDoc
	 */
	public function getDiscussionCountPhrase() {
		return WCF::getLanguage()->getDynamicVariable('wbb.thread.article.replies', ['replies' => ($this->getThread() ? $this->getThread()->replies : 0)]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function renderDiscussions() {
		$thread = $this->getThread();
		if ($thread === null || !$thread->canRead()) {
			return '';
		}
		
		$postList = new ViewablePostList();
		$postList->enableThreadLoading(false);
		$postList->enableModificationLogLoading(false);
		$postList->getConditionBuilder()->add("post.threadID = ?", [$thread->threadID]);
		$postList->getConditionBuilder()->add("post.postID <> ?", [$thread->firstPostID]);
		$postList->getConditionBuilder()->add("post.isDeleted = ? AND post.isDisabled = ?", [0, 0]);
		if ($thread->getBoard()->getPostSortOrder() == 'DESC') {
			$postList->sqlOrderBy = "post.time DESC, post.postID DESC";
		}
		else {
			$postList->sqlOrderBy = "post.time, post.postID";
		}
		$postList->sqlLimit = 10;
		$postList->readObjects();
		
		$likeData = [];
		if (MODULE_LIKE) {
			$postIDs = [];
			foreach ($postList as $post) {
				$postIDs[] = $post->postID;
			}
			$objectType = ReactionHandler::getInstance()->getObjectType('com.woltlab.wbb.likeablePost');
			ReactionHandler::getInstance()->loadLikeObjects($objectType, $postIDs);
			$likeData = ReactionHandler::getInstance()->getLikeObjects($objectType);
		}
		
		WCF::getTPL()->assign([
			'likeData' => $likeData,
			'moreReplies' => $thread->replies - count($postList),
			'posts' => $postList,
			'thread' => new ViewableThread($thread),
		]);
		
		return WCF::getTPL()->fetch('articleReplies', 'wbb');
	}
	
	/**
	 * @inheritDoc
	 */
	public static function isResponsible(Article $article) {
		if ($article->getArticleContent() !== null) {
			/** @noinspection PhpUndefinedFieldInspection */
			return !!$article->getArticleContent()->articleThreadID;
		}
		
		return false;
	}
	
	/**
	 * Returns the discussion thread from the runtime cache.
	 * 
	 * @return      Thread|null
	 */
	protected function getThread() {
		/** @noinspection PhpUndefinedFieldInspection */
		return ThreadRuntimeCache::getInstance()->getObject($this->article->getArticleContent()->articleThreadID);
	}
}
