<?php
namespace wbb\page;
use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\thread\FeedThreadList;
use wcf\page\AbstractFeedPage;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\WCF;

/**
 * Shows threads for the specified boards in feed.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Page
 */
class BoardFeedPage extends AbstractFeedPage {
	/**
	 * list of private board ids
	 * @var integer[]
	 */
	protected $privateBoardIDs = [];
	
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		parent::readParameters();
		
		$permissions = ['canViewBoard', 'canEnterBoard', 'canReadThread'];
		
		if (WCF::getUser()->userID) {
			$this->privateBoardIDs = Board::getPrivateBoardIDs($permissions);
		}
		
		if (empty($this->objectIDs)) {
			// get all board ids
			$this->objectIDs = Board::getAccessibleBoardIDs($permissions);
			if (empty($this->objectIDs) && empty($this->privateBoardIDs)) {
				throw new IllegalLinkException();
			}
		}
		else {
			// validate ids
			$objectIDs = $privateBoardIDs = [];
			foreach ($this->objectIDs as $objectID) {
				$board = BoardCache::getInstance()->getBoard($objectID);
				
				// wrong ID
				if ($board === null) throw new IllegalLinkException();
				
				// guests can never access private boards
				if (!WCF::getUser()->userID && $board->isPrivate) throw new PermissionDeniedException();
				
				// one may not enter
				if (!$board->canEnter()) throw new PermissionDeniedException();
				
				// category or link
				if (!$board->isBoard()) throw new IllegalLinkException();
				
				// threads are not accessible
				if (!$board->getPermission('canReadThread')) throw new PermissionDeniedException();
				
				if (in_array($objectID, $this->privateBoardIDs)) {
					$privateBoardIDs[] = $objectID;
				}
				else {
					$objectIDs[] = $objectID;
				}
			}
			
			$this->objectIDs = $objectIDs;
			$this->privateBoardIDs = $privateBoardIDs;
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		// read the threads
		$this->items = new FeedThreadList($this->objectIDs, $this->privateBoardIDs);
		$this->items->sqlLimit = 20;
		$this->items->readObjects();
		
		// set title
		if (count($this->objectIDs) === 1) {
			$this->title = WCF::getLanguage()->get('wbb.thread.threads') . ' - ' . BoardCache::getInstance()->getBoard(reset($this->objectIDs))->getTitle();
		}
		else {
			$this->title = WCF::getLanguage()->get('wbb.thread.threads');
		}
	}
}

