<?php
namespace wbb\page;
use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\post\Post;
use wbb\data\post\ThreadPostList;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadAction;
use wbb\data\thread\ThreadEditor;
use wbb\data\thread\ViewableThread;
use wbb\system\WBBCore;
use wcf\data\tag\Tag;
use wcf\page\MultipleLinkPage;
use wcf\system\bbcode\BBCodeHandler;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\language\LanguageFactory;
use wcf\system\request\LinkHandler;
use wcf\system\tagging\TagEngine;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\WCF;
use wcf\util\HeaderUtil;

/**
 * Abstract implementation of the thread page.
 *
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Page
 * @since       5.2
 *
 * @property	ThreadPostList		$objectList
 */
class AbstractThreadPage extends MultipleLinkPage implements IBoardPage, IThreadPage {
	/**
	 * @inheritDoc
	 */
	public $itemsPerPage = WBB_THREAD_POSTS_PER_PAGE;
	
	/**
	 * @inheritDoc
	 */
	public $sortOrder = WBB_THREAD_DEFAULT_POST_SORT_ORDER;
	
	/**
	 * thread id
	 * @var	integer
	 */
	public $threadID = 0;
	
	/**
	 * thread object
	 * @var	ViewableThread
	 */
	public $thread;
	
	/**
	 * post id
	 * @var	integer
	 */
	public $postID = 0;
	
	/**
	 * post object
	 * @var	Post
	 */
	public $post;
	
	/**
	 * board object
	 * @var	Board
	 */
	public $board;
	
	/**
	 * list of tags
	 * @var	Tag[]
	 */
	public $tags = [];
	
	/**
	 * number of posts that have been hidden for guests
	 * @var integer
	 */
	public $numberOfHiddenPosts = 0;
	
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		parent::readParameters();
		
		if (isset($_REQUEST['id'])) $this->threadID = intval($_REQUEST['id']);
		if (isset($_REQUEST['postID'])) $this->postID = intval($_REQUEST['postID']);
		if ($this->postID) {
			$this->post = new Post($this->postID);
			if (!$this->post->postID) {
				throw new IllegalLinkException();
			}
			$this->threadID = $this->post->threadID;
		}
		
		$this->thread = ViewableThread::getThread($this->threadID);
		if ($this->thread === null) {
			throw new IllegalLinkException();
		}
		
		// redirect moved threads
		if ($this->thread->movedThreadID) {
			$movedThread = new Thread($this->thread->movedThreadID);
			
			HeaderUtil::redirect($movedThread->getLink(), true, false);
			exit;
		}
		
		if (!$this->thread->canRead()) {
			throw new PermissionDeniedException();
		}
		
		$this->board = BoardCache::getInstance()->getBoard($this->thread->boardID);
		$this->sortOrder = $this->board->getPostSortOrder();
		
		// posts per page
		if ($this->board->postsPerPage) $this->itemsPerPage = $this->board->postsPerPage;
		/** @noinspection PhpUndefinedFieldInspection */
		if (WCF::getUser()->postsPerPage) {
			/** @noinspection PhpUndefinedFieldInspection */
			$this->itemsPerPage = WCF::getUser()->postsPerPage;
		}
		
		// init board style
		$this->board->initStyle();
		
		$this->setCanonicalURL();
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	protected function initObjectList() {
		$this->objectList = new ThreadPostList($this->thread->getDecoratedObject());
		if ($this->sortOrder == 'DESC') $this->objectList->sqlOrderBy = 'post.time DESC, post.postID DESC';
		
		// handle jump to
		if ($this->action == 'lastPost') $this->goToLastPost();
		if ($this->action == 'firstNew') $this->goToFirstNewPost();
		if ($this->postID) $this->goToPost();
	}
	
	/**
	 * @inheritDoc
	 */
	public function calculateNumberOfPages() {
		parent::calculateNumberOfPages();
		
		if (WBB_THREAD_GUEST_MAX_POSTS && !WCF::getUser()->userID) {
			if ($this->items > WBB_THREAD_GUEST_MAX_POSTS) {
				$this->numberOfHiddenPosts = $this->items - WBB_THREAD_GUEST_MAX_POSTS;
				$this->items = WBB_THREAD_GUEST_MAX_POSTS;
				$this->pageNo = 1;
				$this->pages = 1;
				$this->itemsPerPage = WBB_THREAD_GUEST_MAX_POSTS;
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		// add breadcrumbs
		WBBCore::getInstance()->setLocation($this->board->getParentBoards(), $this->board);
		
		$this->increaseViewCounter();
		$this->updateThreadVisit();
		
		// set attachment permissions
		if ($this->objectList->getAttachmentList() !== null) {
			$this->objectList->getAttachmentList()->setPermissions([
				'canDownload' => $this->board->getPermission('canDownloadAttachment'),
				'canViewPreview' => $this->board->getPermission('canViewAttachmentPreview')
			]);
		}
		
		$this->readTags();
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign([
			'attachmentList' => $this->objectList->getAttachmentList(),
			'board' => $this->board,
			'isLastPage' => $this->isLastPage(),
			'sortOrder' => $this->sortOrder,
			'tags' => $this->tags,
			'thread' => $this->thread,
			'threadID' => $this->threadID,
			'numberOfHiddenPosts' => $this->numberOfHiddenPosts
		]);
		
		BBCodeHandler::getInstance()->setDisallowedBBCodes(explode(',', WCF::getSession()->getPermission('user.message.disallowedBBCodes')));
	}
	
	/**
	 * Calculates the position of a specific post in this thread.
	 */
	protected function goToPost() {
		$conditionBuilder = clone $this->objectList->getConditionBuilder();
		$operator = $this->sortOrder == 'ASC' ? '<' : '>';
		// either time must be strictly smaller / greater
		// or time must be equal and postID must be smaller / greater
		// or postID must be equal
		$conditionBuilder->add('(time '.$operator.' ? OR (time = ? AND postID '.$operator.' ?) OR postID = ?)', [$this->post->time, $this->post->time, $this->post->postID, $this->post->postID]);
		
		$sql = "SELECT	COUNT(*) AS posts
			FROM	wbb".WCF_N."_post post
			".$conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		$row = $statement->fetchArray();
		$this->pageNo = intval(ceil($row['posts'] / $this->itemsPerPage));
		
		if ($this->pageNo > 1) {
			$this->setCanonicalURL();
		}
	}
	
	/**
	 * Forwards the user to the last post of the thread.
	 */
	protected function goToLastPost() {
		$sql = "SELECT		post.postID
			FROM		wbb".WCF_N."_post post
			".$this->objectList->getConditionBuilder()."
			ORDER BY	time DESC, postID DESC";
		$statement = WCF::getDB()->prepareStatement($sql, 1);
		$statement->execute($this->objectList->getConditionBuilder()->getParameters());
		$postID = $statement->fetchSingleColumn();
		
		HeaderUtil::redirect(LinkHandler::getInstance()->getLink('Thread', [
			'application' => 'wbb',
			'encodeTitle' => true,
			'object' => $this->thread,
			'postID' => $postID,
		], '#post' . $postID));
		exit;
	}
	
	/**
	 * Forwards the user to the first new post in this thread.
	 */
	protected function goToFirstNewPost() {
		$conditionBuilder = clone $this->objectList->getConditionBuilder();
		$conditionBuilder->add('time > ?', [$this->thread->getVisitTime()]);
		
		$sql = "SELECT		post.postID
			FROM		wbb".WCF_N."_post post
			".$conditionBuilder."
			ORDER BY	time ASC, postID ASC";
		$statement = WCF::getDB()->prepareStatement($sql, 1);
		$statement->execute($conditionBuilder->getParameters());
		$postID = $statement->fetchSingleColumn();
		if ($postID !== false) {
			if ($postID == $this->thread->firstPostID && $this->sortOrder == 'ASC') {
				HeaderUtil::redirect(LinkHandler::getInstance()->getLink('Thread', [
					'application' => 'wbb',
					'encodeTitle' => true,
					'object' => $this->thread
				]));
			}
			else {
				HeaderUtil::redirect(LinkHandler::getInstance()->getLink('Thread', [
					'application' => 'wbb',
					'encodeTitle' => true,
					'object' => $this->thread,
					'postID' => $postID,
				], '#post' . $postID));
			}
			exit;
		}
		else {
			$this->goToLastPost();
		}
	}
	
	/**
	 * Generates the canonical url for the requested page.
	 */
	protected function setCanonicalURL() {
		$this->canonicalURL = LinkHandler::getInstance()->getLink('Thread', [
			'application' => 'wbb',
			'object' => $this->thread
		], ($this->pageNo > 1 ? 'pageNo=' . $this->pageNo : ''));
	}
	
	/**
	 * Increases the view counter of this thread.
	 */
	protected function increaseViewCounter() {
		$threadEditor = new ThreadEditor($this->thread->getDecoratedObject());
		$threadEditor->updateCounters([
			'views' => 1
		]);
	}
	
	/**
	 * Updates the thread visit state for the active user.
	 */
	protected function updateThreadVisit() {
		if ($this->thread->isNew() && $this->objectList->getMaxPostTime() > $this->thread->getVisitTime()) {
			$threadAction = new ThreadAction([$this->thread->getDecoratedObject()], 'markAsRead', [
				'markLikeNotificationsAsConfirmed' => (WCF::getUser()->userID != 0),
				'markModerationNotificationsAsConfirmed' => (WCF::getUser()->userID != 0),
				'visitTime' => $this->objectList->getMaxPostTime(),
				'viewableThread' => $this->thread
			]);
			$threadAction->executeAction();
		}
		else if (WCF::getUser()->userID && UserNotificationHandler::getInstance()->getNotificationCount()) {
			$threadAction = new ThreadAction([$this->thread->getDecoratedObject()], 'markLikeNotificationsAsConfirmed');
			$threadAction->executeAction();
			
			$threadAction = new ThreadAction([$this->thread->getDecoratedObject()], 'markModerationNotificationsAsConfirmed');
			$threadAction->executeAction();
		}
	}
	
	/**
	 * Gets the tags for this thread.
	 */
	protected function readTags() {
		if (MODULE_TAGGING && WBB_THREAD_ENABLE_TAGS && WCF::getSession()->getPermission('user.tag.canViewTag')) {
			$this->tags = TagEngine::getInstance()->getObjectTags(
				'com.woltlab.wbb.thread',
				$this->thread->threadID,
				[$this->thread->languageID === null ? LanguageFactory::getInstance()->getDefaultLanguageID() : ""]
			);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getBoard() {
		return $this->board;
	}
	
	/**
	 * @see	IThreadPage::getThread()
	 * @since	5.0
	 */
	public function getThread() {
		return $this->thread->getDecoratedObject();
	}
}
