<?php
namespace wbb\data\thread\form\option;
use wbb\data\thread\form\ThreadForm;
use wcf\data\custom\option\CustomOptionAction;
use wcf\data\ISortableAction;
use wcf\data\package\PackageCache;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\language\I18nHandler;
use wcf\system\language\LanguageFactory;
use wcf\system\Regex;
use wcf\system\WCF;
use wcf\util\StringUtil;

/**
 * Executes contact option related actions.
 * 
 * @author	Joshua Ruesweg
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Thread\Form\Option
 * @since	5.2
 *
 * @method	ThreadFormOptionEditor[]	getObjects()
 * @method	ThreadFormOptionEditor	        getSingleObject()
 */
class ThreadFormOptionAction extends CustomOptionAction implements ISortableAction {
	/**
	 * @inheritDoc
	 */
	protected $className = ThreadFormOptionEditor::class;
	
	/**
	 * @inheritDoc
	 */
	protected $permissionsCreate = ['admin.board.canAddBoard', 'admin.board.canEditBoard'];
	
	/**
	 * @inheritDoc
	 */
	protected $permissionsDelete = ['admin.board.canAddBoard', 'admin.board.canEditBoard'];
	
	/**
	 * @inheritDoc
	 */
	protected $permissionsUpdate = ['admin.board.canAddBoard', 'admin.board.canEditBoard'];
	
	/**
	 * @inheritDoc
	 */
	protected $resetCache = ['create', 'delete', 'toggle', 'update', 'updatePosition', 'saveEdit'];
	
	/**
	 * @var string
	 */
	protected $optionTitleI18n;
	
	/**
	 * @var string
	 */
	protected $optionDescriptionI18n;
	
	/**
	 * Additional values for the prepareEdit method.
	 * @var array 
	 */
	public $additionalPrepareEditValues = [];
	
	/**
	 * The available option types.
	 * @var string[]
	 */
	public static $availableOptionTypes = [
		'boolean',
		'checkboxes',
		'date',
		'integer',
		'float',
		'multiSelect',
		'radioButton',
		'select',
		'text',
		'textarea',
		'URL',
	];
	
	/**
	 * Options which using select options.
	 * @var string[]
	 */
	public static $optionTypesUsingSelectOptions = [
		'checkboxes',
		'multiSelect',
		'radioButton',
		'select',
	];
	
	/**
	 * Valid data options for updating/creating a option.
	 * @var string[]
	 */
	public static $validDataOptions = [
		'optionTitle',
		'optionTitle_i18n',
		'optionDescription',
		'optionDescription_i18n',
		'optionType',
		'defaultValue',
		'validationPattern',
		'selectOptions',
		'required',
		'showOrder',
		'isDisabled',
		'boardID',
		'formID',
		'tmpHash',
	];
	
	/**
	 * @inheritDoc
	 */
	public function validateUpdate() {
		parent::validateUpdate();
		
		if (!WCF::getSession()->getPermission('admin.board.canEditBoard')) {
			foreach ($this->getObjects() as $object) {
				if ($object->boardID) {
					throw new PermissionDeniedException();
				}
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateDelete() {
		parent::validateDelete();
		
		if (!WCF::getSession()->getPermission('admin.board.canEditBoard')) {
			foreach ($this->getObjects() as $object) {
				if ($object->boardID) {
					throw new PermissionDeniedException();
				}
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateCreate() {
		parent::validateCreate();
		
		if (isset($this->parameters['data']) && is_array($this->parameters['data'])) {
			if (count(array_diff(array_keys($this->parameters['data']), self::$validDataOptions)) !== 0) {
				throw new UserInputException('data');
			}
		}
		else {
			throw new UserInputException('data');
		}
		
		if (!isset($this->parameters['data']['optionTitle'])) {
			if (isset($this->parameters['data']['optionTitle_i18n']) && is_array($this->parameters['data']['optionTitle_i18n'])) {
				$this->optionTitleI18n = $this->parameters['data']['optionTitle_i18n'];
				unset($this->parameters['data']['optionTitle_i18n']);
			}
			else {
				throw new UserInputException('optionTitle');
			}
		}
		else {
			unset($this->parameters['data']['optionTitle_i18n']);
		}
		
		if (!isset($this->parameters['data']['optionDescription'])) {
			if (isset($this->parameters['data']['optionDescription_i18n']) && is_array($this->parameters['data']['optionDescription_i18n'])) {
				$this->optionDescriptionI18n = $this->parameters['data']['optionDescription_i18n'];
				unset($this->parameters['data']['optionDescription_i18n']);
			}
		}
		else {
			unset($this->parameters['data']['optionDescription_i18n']);
		}
		
		$this->parameters['data']['timeCreated'] = TIME_NOW; 
		
		if (isset($this->parameters['data']['formID'])) {
			if ($this->parameters['data']['formID']) {
				$threadForm = new ThreadForm($this->parameters['data']['formID']);
				
				if (!$threadForm->formID) {
					throw new UserInputException('formID');
				}
				
				unset($this->parameters['data']['tmpHash']);
			}
			else {
				unset($this->parameters['data']['formID']);
			}
		}
		
		if (!in_array($this->parameters['data']['optionType'], self::$availableOptionTypes)) {
			throw new UserInputException('optionType', 'invalid');
		}
		
		if (!in_array($this->parameters['data']['optionType'], self::$optionTypesUsingSelectOptions)) {
			$this->parameters['data']['selectOptions'] = '';
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function create() {
		$showOrder = 0;
		if (isset($this->parameters['data']['showOrder'])) {
			$showOrder = $this->parameters['data']['showOrder'];
			unset($this->parameters['data']['showOrder']);
		}
		
		$option = parent::create();
		
		$updateValues = [];
		
		if (!empty($this->optionTitleI18n)) {
			I18nHandler::getInstance()->save($this->optionTitleI18n, 'wbb.thread.threadForm.optionTitle'. $option->optionID, 'wbb.thread', PackageCache::getInstance()->getPackageID('com.woltlab.wbb'));
			$updateValues['optionTitle'] = 'wbb.thread.threadForm.optionTitle'. $option->optionID;
		}
		
		if (!empty($this->optionDescriptionI18n)) {
			I18nHandler::getInstance()->save($this->optionDescriptionI18n, 'wbb.thread.threadForm.optionDescription'. $option->optionID, 'wbb.thread', PackageCache::getInstance()->getPackageID('com.woltlab.wbb'));
			$updateValues['optionDescription'] = 'wbb.thread.threadForm.optionDescription'. $option->optionID;
		}
		
		$optionEditor = new ThreadFormOptionEditor($option);
		if (!empty($updateValues)) {
			$optionEditor->update($updateValues);
		}
		$option = new ThreadFormOption($option->optionID);
		
		$conditionBuilder = new PreparedStatementConditionBuilder();
		
		if ($option->formID) {
			$conditionBuilder->add('formID = ?', [$option->formID]);
		}
		else {
			$conditionBuilder->add('tmpHash = ?', [$option->tmpHash]);
		}
		
		$optionEditor->setShowOrder($showOrder, $conditionBuilder);
		
		return [
			'optionID' => $option->optionID,
			'optionTitle' => WCF::getLanguage()->get($option->optionTitle)
		];
	}
	
	/**
	 * Validates the getDialogAddForm method. 
	 */
	public function validateGetDialogAddForm() {
		WCF::getSession()->checkPermissions(['admin.board.canAddBoard', 'admin.board.canEditBoard']);
	}
	
	/**
	 * Get the thread form add dialog. 
	 * 
	 * @return array
	 */
	public function getDialogAddForm() {
		$languageI18nDummy = [];
		foreach (LanguageFactory::getInstance()->getLanguages() as $language) {
			$languageI18nDummy[$language->languageID] = '';
		}
		
		WCF::getTPL()->assign([
			'defaultValue' => '',
			'validationPattern' => '',
			'optionType' => 'text',
			'selectOptions' => '',
			'required' => 0,
			'showOrder' => 0,
			'availableOptionTypes' => self::$availableOptionTypes,
			'optionTypesUsingSelectOptions' => self::$optionTypesUsingSelectOptions,
			'availableLanguages' => LanguageFactory::getInstance()->getLanguages(),
			'i18nPlainValues' => [
				'optionTitle' => '',
				'optionDescription' => '',
			],
			'i18nValues' => [
				'optionTitle' => $languageI18nDummy,
				'optionDescription' => $languageI18nDummy,
			],
			'errorField' => '',
			'isDisabled' => false,
		]);
		
		return [
			'template' => WCF::getTPL()->fetch('customOptionAdd')
		];
	}
	
	/**
	 * Validate the prepareEdit method. 
	 */
	public function validatePrepareEdit() {
		$this->validateUpdate();
		
		if (count($this->objects) > 1) {
			throw new UserInputException('objects');
		}
	}
	
	/**
	 * Returns the data for the edit form for a specific thread form option. 
	 * 
	 * @return array
	 */
	public function prepareEdit() {
		/** @var ThreadFormOption $option */
		$option = reset($this->objects);
		
		$data = [
			'simpleValues' => [
				'defaultValue' => $option->defaultValue,
				'validationPattern' => $option->validationPattern,
				'selectOptions' => $option->selectOptions,
				'showOrder' => $option->showOrder,
			],
			'checkboxValues' => [
				'required' => $option->required, 
				'isDisabled' => $option->isDisabled,
			],
			'selectOption' => [
				'optionType' => $option->optionType,
			],
			'i18nValues' => [
				'optionTitle' => $this->getI18n('wbb.thread.threadForm.optionTitle\d+', $option->optionTitle),
				'optionDescription' => $this->getI18n('wbb.thread.threadForm.optionDescription\d+', $option->optionDescription),
			],
		];
		
		return array_merge($data, $this->additionalPrepareEditValues);
	}
	
	/**
	 * Validates the saveEdit method
	 */
	public function validateSaveEdit() {
		$this->validateUpdate();
		
		if (isset($this->parameters['data']) && is_array($this->parameters['data'])) {
			if (count(array_diff(array_keys($this->parameters['data']), self::$validDataOptions)) !== 0) {
				throw new UserInputException('data');
			}
		}
		else {
			throw new UserInputException('data');
		}
		
		if (!isset($this->parameters['data']['optionTitle'])) {
			if (isset($this->parameters['data']['optionTitle_i18n']) && is_array($this->parameters['data']['optionTitle_i18n'])) {
				$this->optionTitleI18n = $this->parameters['data']['optionTitle_i18n'];
				unset($this->parameters['data']['optionTitle_i18n']);
			}
			else {
				throw new UserInputException('optionTitle');
			}
		}
		else {
			unset($this->parameters['data']['optionTitle_i18n']);
		}
		
		if (!isset($this->parameters['data']['optionDescription'])) {
			if (isset($this->parameters['data']['optionDescription_i18n']) && is_array($this->parameters['data']['optionDescription_i18n'])) {
				$this->optionDescriptionI18n = $this->parameters['data']['optionDescription_i18n'];
				unset($this->parameters['data']['optionDescription_i18n']);
			}
		}
		else {
			unset($this->parameters['data']['optionDescription_i18n']);
		}
		
		if (isset($this->parameters['data']['tmpHash'])) {
			throw new UserInputException('tmpHash', 'noValidUpdateValue');
		}
		
		if (isset($this->parameters['data']['boardID'])) {
			throw new UserInputException('boardID', 'noValidUpdateValue');
		}
		
		if (!in_array($this->parameters['data']['optionType'], self::$availableOptionTypes)) {
			throw new UserInputException('optionType', 'invalid');
		}
		
		if (!in_array($this->parameters['data']['optionType'], self::$optionTypesUsingSelectOptions)) {
			$this->parameters['data']['selectOptions'] = '';
		}
	}
	
	/**
	 * Updates a thread form option.
	 */
	public function saveEdit() {
		$this->update();
		
		foreach ($this->getObjects() as $object) {
			$updateValues = [];
			
			if (!empty($this->optionTitleI18n)) {
				I18nHandler::getInstance()->save($this->optionTitleI18n, 'wbb.thread.threadForm.optionTitle'. $object->optionID, 'wbb.thread', PackageCache::getInstance()->getPackageID('com.woltlab.wbb'));
				$updateValues['optionTitle'] = 'wbb.thread.threadForm.optionTitle'. $object->optionID;
			}
			
			if (!empty($this->optionDescriptionI18n)) {
				I18nHandler::getInstance()->save($this->optionDescriptionI18n, 'wbb.thread.threadForm.optionDescription'. $object->optionID, 'wbb.thread', PackageCache::getInstance()->getPackageID('com.woltlab.wbb'));
				$updateValues['optionDescription'] = 'wbb.thread.threadForm.optionDescription'. $object->optionID;
			}
			
			if (!empty($updateValues)) {
				$object->update($updateValues);
			}
		}
	}
	
	/**
	 * Returns the i18n value for a field. 
	 * 
	 * @param       string          $pattern
	 * @param       string          $plainValue
	 * @return      string[]
	 */
	private function getI18n($pattern, $plainValue) {
		$isI18n = Regex::compile('^'.$pattern.'$')->match($plainValue);
		if (!$isI18n) {
			// check if it's a regular language variable
			$isI18n = Regex::compile('^([a-zA-Z0-9-_]+\.)+[a-zA-Z0-9-_]+$')->match($plainValue);
		}
		
		if ($isI18n) {
			// use i18n values from language items
			$sql = "SELECT	        languageID, languageItemValue
				FROM	        wcf".WCF_N."_language_item
				WHERE	        languageItem = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([
				$plainValue
			]);
			$i18nValues = [];
			while ($row = $statement->fetchArray()) {
				$languageItemValue = StringUtil::unifyNewlines($row['languageItemValue']);
				$i18nValues[$row['languageID']] = StringUtil::encodeJS($languageItemValue);
				
				if ($row['languageID'] == LanguageFactory::getInstance()->getDefaultLanguageID()) {
					$value = $languageItemValue;
				}
			}
			
			// item appeared to be a language item but either is not or does not exist
			if (empty($i18nValues) && empty($value)) {
				return [
					0 => $plainValue
				];
			}
			else {
				return $i18nValues;
			}
		}
		else {
			return [
				0 => $plainValue
			];
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateUpdatePosition() {
		WCF::getSession()->checkPermissions($this->permissionsUpdate);
		
		if (!isset($this->parameters['optionIDs']) || !is_array($this->parameters['optionIDs'])) {
			throw new UserInputException('optionIDs');
		}
		
		$threadFormOptionList = new ThreadFormOptionList();
		$threadFormOptionList->setObjectIDs($this->parameters['optionIDs']);
		$threadFormOptionList->readObjects();
		if (count($threadFormOptionList->getObjects()) != count($this->parameters['optionIDs'])) {
			throw new UserInputException('optionIDs', 'invalid');
		}
	}
	
	/**
	 * Copies form values from a post to another post.
	 */
	public function copy() {
		$oldPostID = intval($this->parameters['oldPostID']);
		$newPostID = intval($this->parameters['newPostID']);
		
		$sql = "INSERT INTO     wbb". WCF_N ."_thread_form_option_value
                        SELECT          ? AS postID, optionID, optionValue 
                        FROM            wbb". WCF_N ."_thread_form_option_value 
                        WHERE           postID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$newPostID, $oldPostID]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function updatePosition() {
		$sql = "UPDATE	wbb".WCF_N."_thread_form_option
			SET	showOrder = ?
			WHERE	optionID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		
		$showOrder = 0;
		WCF::getDB()->beginTransaction();
		foreach ($this->parameters['optionIDs'] as $optionID) {
			$statement->execute([
				$showOrder++,
				$optionID,
			]);
		}
		WCF::getDB()->commitTransaction();
	}
}
