<?php
namespace wbb\data\thread;
use wbb\data\post\ViewablePostList;
use wbb\system\label\object\ThreadLabelObjectHandler;
use wcf\data\moderation\queue\ModerationQueue;
use wcf\data\object\type\ObjectTypeCache;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\reaction\ReactionHandler;
use wcf\system\visitTracker\VisitTracker;
use wcf\system\WCF;

/**
 * Represents a list of threads.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Thread
 *
 * @method	ViewableThread		current()
 * @method	ViewableThread[]	getObjects()
 * @method	ViewableThread|null	search($objectID)
 * @property	ViewableThread[]	$objects
 */
class ViewableThreadList extends ThreadList {
	/**
	 * @inheritDoc
	 */
	public $sqlOrderBy = 'thread.lastPostTime DESC';
	
	/**
	 * @inheritDoc
	 */
	public $decoratorClassName = ViewableThread::class;
	
	/**
	 * if `true`, first posts of the threads are loaded
	 * @var	boolean
	 */
	public $firstPostLoading = false;
	
	/**
	 * @inheritDoc
	 */
	public function __construct($loadLikes = true, $loadThreadsOnly = false) {
		parent::__construct();
		
		if ($loadThreadsOnly) {
			// skip other joins, greatly improves performance on board list
			return;
		}
		
		if (WCF::getUser()->userID) {
			// own posts
			if (!empty($this->sqlSelects)) $this->sqlSelects .= ',';
			$this->sqlSelects = "DISTINCT post.userID AS ownPosts";
			$this->sqlJoins = " LEFT JOIN wbb".WCF_N."_post post ON (post.threadID = thread.threadID AND post.userID = ".WCF::getUser()->userID.")";
			
			// last visit time
			if (!empty($this->sqlSelects)) $this->sqlSelects .= ',';
			$this->sqlSelects .= 'tracked_visit.visitTime';
			$this->sqlJoins .= " LEFT JOIN wcf".WCF_N."_tracked_visit tracked_visit ON (tracked_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.thread')." AND tracked_visit.objectID = thread.threadID AND tracked_visit.userID = ".WCF::getUser()->userID.")";
			
			if (!empty($this->sqlSelects)) $this->sqlSelects .= ',';
			$this->sqlSelects .= 'tracked_board_visit.visitTime AS boardVisitTime';
			$this->sqlJoins .= " LEFT JOIN wcf".WCF_N."_tracked_visit tracked_board_visit ON (tracked_board_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.board')." AND tracked_board_visit.objectID = thread.boardID AND tracked_board_visit.userID = ".WCF::getUser()->userID.")";
			
			// subscriptions
			if (!empty($this->sqlSelects)) $this->sqlSelects .= ',';
			$this->sqlSelects .= 'user_object_watch.watchID, user_object_watch.notification';
			$this->sqlJoins .= " LEFT JOIN wcf".WCF_N."_user_object_watch user_object_watch ON (user_object_watch.objectTypeID = ".ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.user.objectWatch', 'com.woltlab.wbb.thread')->objectTypeID." AND user_object_watch.userID = ".WCF::getUser()->userID." AND user_object_watch.objectID = thread.threadID)";
		}
		
		if ($loadLikes) {
			if (!empty($this->sqlSelects)) $this->sqlSelects .= ',';
			$this->sqlSelects .= "like_object.cachedReactions";
			$this->sqlJoins .= " LEFT JOIN wcf" . WCF_N . "_like_object like_object ON (like_object.objectTypeID = " . ReactionHandler::getInstance()->getObjectType('com.woltlab.wbb.likeablePost')->objectTypeID . " AND like_object.objectID = thread.firstPostID)";
		}
		
		// get report status
		if (WCF::getSession()->getPermission('mod.general.canUseModeration')) {
			if (!empty($this->sqlSelects)) $this->sqlSelects .= ',';
			$this->sqlSelects .= "moderation_queue.queueID AS reportQueueID";
			$this->sqlJoins .= " LEFT JOIN wcf".WCF_N."_moderation_queue moderation_queue ON (moderation_queue.objectTypeID = ".ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.moderation.report', 'com.woltlab.wbb.post')." AND moderation_queue.objectID = thread.firstPostID AND moderation_queue.status <> ".ModerationQueue::STATUS_DONE." AND moderation_queue.status <> ".ModerationQueue::STATUS_REJECTED." AND moderation_queue.status <> ".ModerationQueue::STATUS_CONFIRMED.")";
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readObjects() {
		if ($this->objectIDs === null) $this->readObjectIDs();
		parent::readObjects();
		
		// get assigned labels
		$firstPostIDs = $threadIDs = $userIDs = [];
		
		/** @var Thread $thread */
		foreach ($this->objects as $thread) {
			if ($thread->hasLabels) {
				$threadIDs[] = $thread->threadID;
			}
			
			if ($thread->userID) {
				$userIDs[] = $thread->userID;
			}
			
			if ($thread->lastPosterID) {
				$userIDs[] = $thread->lastPosterID;
			}
			
			if ($this->firstPostLoading && $thread->firstPostID) {
				$firstPostIDs[] = $thread->firstPostID;
			}
		}
		
		if (!empty($threadIDs)) {
			$assignedLabels = ThreadLabelObjectHandler::getInstance()->getAssignedLabels($threadIDs);
			foreach ($assignedLabels as $threadID => $labels) {
				foreach ($labels as $label) {
					$this->objects[$threadID]->addLabel($label);
				}
			}
		}
		
		if (!empty($userIDs)) {
			UserProfileRuntimeCache::getInstance()->cacheObjectIDs(array_unique($userIDs));
		}
		
		if (!empty($firstPostIDs)) {
			$postList = new ViewablePostList();
			$postList->enableEmbeddedObjectLoading();
			$postList->setObjectIDs($firstPostIDs);
			$postList->readObjects();
			
			/** @var Thread $thread */
			foreach ($this->objects as $thread) {
				if ($thread->firstPostID) {
					$thread->setFirstPost($postList->search($thread->firstPostID)->getDecoratedObject());
				}
			}
		}
	}
}
