<?php
namespace wbb\data\thread;
use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\modification\log\ViewableBoardThreadModificationLog;
use wbb\data\post\Post;
use wbb\data\post\ThreadPostList;
use wcf\data\label\Label;
use wcf\data\language\Language;
use wcf\data\user\UserProfile;
use wcf\data\DatabaseObjectDecorator;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\language\LanguageFactory;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\user\UserProfileHandler;
use wcf\system\visitTracker\VisitTracker;
use wcf\system\WCF;

/**
 * Represents a viewable thread.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Thread
 * 
 * @method	Thread		getDecoratedObject()
 * @mixin	Thread
 * @property	integer|null	$threadNewTime
 * @property	integer|null	$notification
 * @property	integer|null	$watchID
 * @property	integer|null	$ownPosts
 * @property	integer|null	$visitTime
 * @property	integer|null	$boardVisitTime
 * @property	integer|null	$likes
 * @property	integer|null	$dislikes
 * @property	integer|null	$reportQueueID
 */
class ViewableThread extends DatabaseObjectDecorator {
	/**
	 * @inheritDoc
	 */
	protected static $baseClass = Thread::class;
	
	/**
	 * user profile object
	 * @var	UserProfile
	 */
	protected $userProfile;
	
	/**
	 * last poster's profile
	 * @var	UserProfile
	 */
	protected $lastPosterProfile;
	
	/**
	 * list of assigned labels
	 * @var	Label[]
	 */
	protected $labels = [];
	
	/**
	 * modification log entry
	 * @var	ViewableBoardThreadModificationLog
	 */
	protected $logEntry;
	
	/**
	 * effective visit time
	 * @var	integer
	 */
	protected $effectiveVisitTime;
	
	/**
	 * number of unread threads
	 * @var	integer
	 */
	protected static $unreadThreads;
	
	/**
	 * maps legacy direct access to last poster's user profile data to the real
	 * user profile property names
	 * @var	string[]
	 * @deprecated
	 */
	protected static $__lastUserAvatarPropertyMapping = [
		'lastPosterAvatarID' => 'avatarID',
		'lastPosterAvatarName' => 'avatarName',
		'lastPosterAvatarExtension' => 'avatarExtension',
		'lastPosterAvatarWidth' => 'width',
		'lastPosterAvatarHeight' => 'height',
		'lastPosterEmail' => 'email',
		'lastPosterDisableAvatar' => 'disableAvatar',
		'lastPosterEnableGravatar' => 'enableGravatar',
		'lastPosterGravatarFileExtension' => 'gravatarFileExtension',
		'lastPosterAvatarFileHash' => 'fileHash'
	];
	
	/**
	 * @inheritDoc
	 * @deprecated	5.0
	 * @since	5.0
	 */
	public function __get($name) {
		$value = parent::__get($name);
		if ($value !== null) {
			return $value;
		}
		else if (array_key_exists($name, $this->object->data)) {
			return null;
		}
		
		/** @noinspection PhpVariableVariableInspection */
		$value = $this->getUserProfile()->$name;
		if ($value !== null) {
			return $value;
		}
		
		if (isset(static::$__lastUserAvatarPropertyMapping[$name])) {
			return $this->getLastPosterProfile()->getAvatar()->{static::$__lastUserAvatarPropertyMapping[$name]};
		}
		
		return null;
	}
	
	/**
	 * Returns the user profile object.
	 * 
	 * @return	UserProfile
	 */
	public function getUserProfile() {
		if ($this->userProfile === null) {
			if ($this->userID) {
				$this->userProfile = UserProfileRuntimeCache::getInstance()->getObject($this->userID);
			}
			else {
				$this->userProfile = UserProfile::getGuestUserProfile($this->username);
			}
		}
		
		return $this->userProfile;
	}
	
	/**
	 * Returns the user profile object.
	 * 
	 * @return	UserProfile
	 */
	public function getLastPosterProfile() {
		if ($this->lastPosterProfile === null) {
			if ($this->lastPosterID) {
				$this->lastPosterProfile = UserProfileRuntimeCache::getInstance()->getObject($this->lastPosterID);
			}
			else {
				$this->lastPosterProfile = UserProfile::getGuestUserProfile($this->lastPoster);
			}
		}
		
		return $this->lastPosterProfile;
	}
	
	/**
	 * Returns the effective visit time.
	 * 
	 * @return	integer
	 */
	public function getVisitTime() {
		if ($this->effectiveVisitTime === null) {
			if (WCF::getUser()->userID) {
				/** @noinspection PhpUndefinedFieldInspection */
				$this->effectiveVisitTime = max($this->visitTime, $this->boardVisitTime, VisitTracker::getInstance()->getVisitTime('com.woltlab.wbb.thread'));
			}
			else {
				$this->effectiveVisitTime = max(VisitTracker::getInstance()->getObjectVisitTime('com.woltlab.wbb.thread', $this->threadID), VisitTracker::getInstance()->getObjectVisitTime('com.woltlab.wbb.board', $this->boardID), VisitTracker::getInstance()->getVisitTime('com.woltlab.wbb.thread'));
			}
			if ($this->effectiveVisitTime === null) {
				$this->effectiveVisitTime = 0;
			}
		}
		
		return $this->effectiveVisitTime;
	}
	
	/**
	 * Returns true if this thread is new for the active user.
	 * 
	 * @return	boolean
	 */
	public function isNew() {
		if (!$this->movedThreadID) {
			// threadNewTime can be set by ThreadList::excludeReadThreads()
			if ($this->threadNewTime) {
				return $this->threadNewTime > $this->getVisitTime();
			}
			
			if ($this->lastPostTime > $this->getVisitTime()) {
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Returns true if the active user doesn't have read the given post.
	 * 
	 * @param	Post	$post
	 * @return	boolean
	 */
	public function isNewPost(Post $post) {
		return ($post->time > $this->getVisitTime());
	}
	
	/**
	 * Returns the number of pages in this thread.
	 * 
	 * @return	integer
	 */
	public function getPages() {
		/** @noinspection PhpUndefinedFieldInspection */
		if (WCF::getUser()->postsPerPage) {
			/** @noinspection PhpUndefinedFieldInspection */
			$postsPerPage = WCF::getUser()->postsPerPage;
		}
		else if (BoardCache::getInstance()->getBoard($this->boardID)->postsPerPage) {
			$postsPerPage = BoardCache::getInstance()->getBoard($this->boardID)->postsPerPage;
		}
		else {
			$postsPerPage = WBB_THREAD_POSTS_PER_PAGE;
		}
		
		return intval(ceil(($this->replies + 1) / $postsPerPage));
	}
	
	/**
	 * Returns the language of this thread.
	 * 
	 * @return	Language
	 */
	public function getLanguage() {
		if ($this->languageID) return LanguageFactory::getInstance()->getLanguage($this->languageID);
		
		return null;
	}
	
	/**
	 * Returns the flag icon for the thread language.
	 * 
	 * @return	string
	 */
	public function getLanguageIcon() {
		return '<img src="'.$this->getLanguage()->getIconPath().'" alt="" class="iconFlag">';
	}
	
	/**
	 * Returns the viewable thread object with the given thread id.
	 * 
	 * @param	integer		$threadID
	 * @return	ViewableThread
	 */
	public static function getThread($threadID) {
		$list = new ViewableThreadList();
		$list->setObjectIDs([$threadID]);
		$list->readObjects();
		$objects = $list->getObjects();
		if (isset($objects[$threadID])) return $objects[$threadID];
		return null;
	}
	
	/**
	 * Adds a label.
	 * 
	 * @param	Label	$label
	 */
	public function addLabel(Label $label) {
		$this->labels[$label->labelID] = $label;
	}
	
	/**
	 * Returns a list of labels.
	 * 
	 * @return	Label[]
	 */
	public function getLabels() {
		return $this->labels;
	}
	
	/**
	 * Returns the primary (first) label.
	 * 
	 * @return	Label|null
	 */
	public function getPrimaryLabel() {
		if (!$this->hasLabels()) return null;
		
		return reset($this->labels);
	}
	
	/**
	 * Returns true if one or more labels are assigned to this thread.
	 * 
	 * @return	boolean
	 */
	public function hasLabels() {
		return count($this->labels) ? true : false;
	}
	
	/**
	 * Sets modification log entry.
	 * 
	 * @param	ViewableBoardThreadModificationLog	$logEntry
	 */
	public function setLogEntry(ViewableBoardThreadModificationLog $logEntry) {
		$this->logEntry = $logEntry;
	}
	
	/**
	 * Returns delete note if applicable.
	 * 
	 * @return	string
	 */
	public function getDeleteNote() {
		if ($this->logEntry === null || $this->logEntry->action != 'trash') {
			return '';
		}
		
		return $this->logEntry->__toString();
	}
	
	/**
	 * @inheritDoc
	 */
	public function isSubscribed() {
		return $this->watchID ? 1 : 0;
	}
	
	/**
	 * Returns last post time for current thread. Please be aware that it might
	 * not return the actual last post time if you're not on the last page!
	 * 
	 * @param	ThreadPostList		$postList
	 * @return	integer
	 */
	public function getLastPostTime(ThreadPostList $postList) {
		$lastPostTime = $this->lastPostTime;
		foreach ($postList as $post) {
			if ($post->time > $lastPostTime) {
				$lastPostTime = $post->time;
			}
		}
		
		return $lastPostTime;
	}
	
	/**
	 * Returns formatted message text.
	 * 
	 * @param int $maxLength
	 * @param bool $removeLinks
	 * @return string
	 */
	public function getFormattedExcerpt($maxLength = WBB_THREAD_MAX_FORMATTED_EXCERPT_LENGTH, $removeLinks = false) {
		return $this->getFirstPost()->getFormattedExcerpt($maxLength, $removeLinks);
	}
	
	/**
	 * Returns the number of unread threads.
	 * 
	 * @return	integer
	 */
	public static function getUnreadThreads() {
		if (self::$unreadThreads === null) {
			self::$unreadThreads = 0;
			
			if (WCF::getUser()->userID) {
				$data = UserStorageHandler::getInstance()->getField('wbbUnreadThreads');
				
				// cache does not exist or is outdated
				if ($data === null) {
					$boardIDs = Board::filterBoardIDs(Board::getAccessibleBoardIDs());
					$privateBoardIDs = Board::filterBoardIDs(Board::getPrivateBoardIDs());
					
					if (!empty($boardIDs) || !empty($privateBoardIDs)) {
						$conditionBuilder = new PreparedStatementConditionBuilder();
						
						if (empty($privateBoardIDs)) {
							$conditionBuilder->add("thread.boardID IN (?)", [$boardIDs]);
						}
						else if (empty($boardIDs)) {
							$conditionBuilder->add("(thread.boardID IN (?) AND thread.userID = ?)", [$privateBoardIDs, WCF::getUser()->userID]);
						}
						else {
							$conditionBuilder->add("(thread.boardID IN (?) OR (thread.boardID IN (?) AND thread.userID = ?))", [$boardIDs, $privateBoardIDs, WCF::getUser()->userID]);
						}
						
						$conditionBuilder->add("thread.lastPostTime > ?", [VisitTracker::getInstance()->getVisitTime('com.woltlab.wbb.thread')]);
						$conditionBuilder->add("thread.isDeleted = 0 AND thread.isDisabled = 0 AND thread.movedThreadID IS NULL");
						$conditionBuilder->add("(thread.lastPostTime > tracked_thread_visit.visitTime OR tracked_thread_visit.visitTime IS NULL)");
						$conditionBuilder->add("(thread.lastPostTime > tracked_board_visit.visitTime OR tracked_board_visit.visitTime IS NULL)");
						// apply language filter
						if (LanguageFactory::getInstance()->multilingualismEnabled() && count(WCF::getUser()->getLanguageIDs())) {
							$conditionBuilder->add('(thread.languageID IN (?) OR thread.languageID IS NULL)', [WCF::getUser()->getLanguageIDs()]);
						}
						
						if (!empty(UserProfileHandler::getInstance()->getIgnoredUsers())) {
							$conditionBuilder->add("(thread.userID IS NULL OR thread.userID NOT IN (?))", [UserProfileHandler::getInstance()->getIgnoredUsers()]);
						}
						
						$sql = "SELECT		COUNT(*) AS count
							FROM		wbb".WCF_N."_thread thread
							LEFT JOIN	wcf".WCF_N."_tracked_visit tracked_thread_visit
							ON		(tracked_thread_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.thread')." AND tracked_thread_visit.objectID = thread.threadID AND tracked_thread_visit.userID = ".WCF::getUser()->userID.")
							LEFT JOIN	wcf".WCF_N."_tracked_visit tracked_board_visit
							ON		(tracked_board_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.board')." AND tracked_board_visit.objectID = thread.boardID AND tracked_board_visit.userID = ".WCF::getUser()->userID.")
							".$conditionBuilder;
						$statement = WCF::getDB()->prepareStatement($sql);
						$statement->execute($conditionBuilder->getParameters());
						$row = $statement->fetchArray();
						self::$unreadThreads = $row['count'];
					}
					
					// update storage data
					UserStorageHandler::getInstance()->update(WCF::getUser()->userID, 'wbbUnreadThreads', self::$unreadThreads);
				}
				else {
					self::$unreadThreads = $data;
				}
			}
		}
		
		return self::$unreadThreads;
	}
}
